% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/dist_edr.R
\name{dist_edr}
\alias{dist_edr}
\title{Dissimilarities between Ecological Dynamic Regimes}
\usage{
dist_edr(
  d,
  d.type,
  trajectories = NULL,
  states = NULL,
  edr,
  metric = "dDR",
  symmetrize = NULL,
  ...
)
}
\arguments{
\item{d}{Symmetric matrix or object of class \code{\link{dist}} containing the
dissimilarities between each pair of states of all trajectories in the EDR or
the dissimilarities between each pair of trajectories.}

\item{d.type}{One of \code{"dStates"} (if \code{d} contains state dissimilarities) or
\code{"dTraj"} (if \code{d} contains trajectory dissimilarities).}

\item{trajectories}{Only if \code{d.type} = \code{"dStates"}. Vector indicating the
trajectory or site corresponding to each entry in \code{d}.}

\item{states}{Only if \code{d.type} = \code{"dStates"}. Vector of integers indicating the
order of the states in \code{d} for each trajectory.}

\item{edr}{Vector indicating the EDR to which each trajectory/state in \code{d} belongs.}

\item{metric}{A string indicating the dissimilarity index to be used: \code{"dDR"}
(default), \code{"minDist"}, \code{"maxDist"}.}

\item{symmetrize}{String naming the function to be called to symmetrize the
resulting dissimilarity matrix (\code{"mean"}, \code{"min"}, \verb{"max}, \code{"lower"}, \code{"upper"}).
If \code{NULL} (default), the matrix is not symmetrized.}

\item{...}{Only if \code{d.type} = \code{"dStates"}. Further arguments to calculate
trajectory dissimilarities. See \code{\link[ecotraj:trajectoryComparison]{ecotraj::trajectoryDistances()}}.}
}
\value{
Matrix including the dissimilarities between every pair of EDRs.
}
\description{
Generate a matrix containing dissimilarities between one or more pairs of
Ecological Dynamic Regimes (EDR). \code{dist_edr()} computes different dissimilarity
indices, all of them based on the dissimilarities between the trajectories of
two EDRs.
}
\details{
The implemented metrics are:

\describe{
\item{\code{"dDR"}}{
\eqn{
d_{DR}(R_1, R_2) = \frac{1}{n} \sum_{i=1}^{n} d_{TR}(T_{1i}, R_2)
}
}

\item{\code{"minDist"}}{
\eqn{
d_{DRmin}(R_1, R_2) = \min_{i=1}^{n} \{ d_{TR}(T_{1i}, R_2) \}
}
}

\item{\code{"maxDist"}}{
\eqn{
d_{DRmax}(R_1, R_2) = \max_{i=1}^{n} \{ d_{TR}(T_{1i}, R_2) \}
}
}
}

where \eqn{R_1} and \eqn{R_2} are two EDRs composed of \eqn{n} and \eqn{m}
ecological trajectories, respectively, and \eqn{d_{TR}(T_{1i}, R_2)} is the
dissimilarity between the trajectory \eqn{T_{1i}} of \eqn{R_1} and the closest
trajectory of \eqn{R_2}:

\eqn{
d_{TR}(T_{1i}, R_2) = \min\{d_T(T_{1i}, T_{21}), ... , d_T(T_{1i}, T_{2m})\}
}

The metrics calculated are not necessarily symmetric. That is, \eqn{d_{DR}(R_1, R_2)}
is not necessarily equal to \eqn{d_{DR}(R_2, R_1)}. It is possible to symmetrize
the returned matrix by indicating the name of the function to be used in \code{symmetrize}:

\describe{
\item{\code{"mean"}}{
\eqn{
d_{DRsym} = \frac{d_{DR}(R_1, R_2) + d_{DR}(R_2, R_1)}{2}
}
}

\item{\code{"min"}}{
\eqn{
d_{DRsym} = \min\{d_{DR}(R_1, R_2), d_{DR}(R_2, R_1)\}
}
}

\item{\code{"max"}}{
\eqn{
d_{DRsym} = \max\{d_{DR}(R_1, R_2), d_{DR}(R_2, R_1)\}
}
}

\item{\code{"lower"}}{
The lower triangular part of the dissimilarity matrix is used.
}

\item{\code{"upper"}}{
The upper triangular part of the dissimilarity matrix is used.
}

}
}
\examples{
# Load species abundances and compile in a data frame
abun1 <- EDR_data$EDR1$abundance
abun2 <- EDR_data$EDR2$abundance
abun3 <- EDR_data$EDR3$abundance
abun <- data.frame(rbind(abun1, abun2, abun3))

# Define row names in abun to keep the reference of the EDR, trajectory, and
# state
row.names(abun) <- paste0(abun$EDR, "_", abun$traj, "_", abun$state)

# Calculate dissimilarities between every pair of states
# For example, Bray-Curtis index
dStates <- vegan::vegdist(abun[, -c(1, 2, 3)], method = "bray")

# Use the labels in dStates to define the trajectories to which each state
# belongs
id_traj <- vapply(strsplit(labels(dStates), "_"), function(x){
                    paste0(x[1], "_", x[2])
                }, character(1))
id_state <- vapply(strsplit(labels(dStates), "_"), function(x){
                    as.integer(x[3])
                }, integer(1))
id_edr <- vapply(strsplit(labels(dStates), "_"), function(x){
                    paste0("EDR", x[1])
                }, character(1))

# Calculate dissimilarities between every pair of trajectories
dTraj <- ecotraj::trajectoryDistances(ecotraj::defineTrajectories(d = dStates, sites = id_traj,
                                                                  surveys = id_state),
                                      distance.type = "DSPD")

# Use labels in dTraj to identify EDRs
id_edr_traj <- vapply(strsplit(labels(dTraj), "_"), function(x){
                    paste0("EDR", x[1])
                }, character(1))

# Compute dissimilarities between EDRs:
# 1) without symmetrizing the matrix and using state dissimilarities
dEDR <- dist_edr(d = dStates, d.type = "dStates",
                 trajectories = id_traj, states = id_state, edr = id_edr,
                 metric = "dDR", symmetrize = NULL)

# 2) symmetrizing by averaging elements on and below the diagonal and using
# trajectory dissimilarities
dEDR <- dist_edr(d = dTraj, d.type = "dTraj", edr = id_edr_traj,
                 metric = "dDR", symmetrize = "mean")


}
\references{
Sánchez-Pinillos, M., Kéfi, S., De Cáceres, M., Dakos, V. 2023. Ecological Dynamic
Regimes: Identification, characterization, and comparison. \emph{Ecological Monographs}.
\url{doi:10.1002/ecm.1589}
}
\author{
Martina Sánchez-Pinillos
}
