#' SAM
#'
#' @description
#' Calculate the size adherence multiplier (SAM), which evaluates adherence to
#' harvest slot limits by comparing (\code{lower}, \code{upper}) percentiles of length–frequency
#' data to \code{minLS} and \code{maxLS}. Optionally, produce a graph showing
#' how SAM varies across a grid of (\code{lower}, \code{upper}) values.
#'
#' When \code{lower >= minLS} and \code{upper <= maxLS}, neither of the slot
#' limits are violated and the multiplier is calculated without constraint.
#' When \code{lower < minLS} or \code{upper > maxLS}, at least one slot limit is
#' violated and the multiplier is capped at \code{constraint} (default = 1).
#'
#' @param lower Numeric (length 1). Lower percentile of catch length (e.g., 2.5th).
#' @param upper Numeric (length 1). Upper percentile of catch length (e.g., 97.5th).
#' @param minLS Numeric (length 1). Minimum landing size (must be > 0).
#' @param maxLS Numeric (length 1). Maximum landing size (must be > 0).
#' @param constraint Numeric (length 1) in \eqn{[0,1]}. Cap applied \emph{when either}
#'   slot limit is violated (default = 1).
#' @param digits Integer. Number of decimal places used to round outputs
#'   (default = 2). Set \code{digits = NA} to prevent rounding.
#' @param plot Logical. If \code{TRUE}, include a \pkg{ggplot2} plot of the calculated value
#'   on a grid of (\code{lower}, \code{upper}) combinations (default \code{FALSE}).
#' @param res Numeric > 0. Grid step for plotting when \code{plot = TRUE}. Smaller
#'   values increase smoothness but can be slower (default \code{1}).
#' @param lower_percentile,upper_percentile Numbers used only for axis labels when
#'   \code{plot = TRUE} (defaults \code{2.5} and \code{97.5}).
#' @param length_units Optional character scalar. Units to display in the x/y-axis
#'   labels when `plot = TRUE` (e.g., `"cm"` or `"mm"`). If `NULL` (default),
#'   units are omitted.
#'
#' @return A list with:
#' \describe{
#'   \item{lower_adherence}{Relative deviation of \code{lower} from \code{minLS}: \eqn{(lower - minLS)/minLS}.}
#'   \item{upper_adherence}{Relative deviation of \code{upper} from \code{maxLS}: \eqn{(maxLS - upper)/maxLS}.}
#'   \item{SAM}{Size adherence multiplier. \code{SAM > 1} increases the advised catch; \code{SAM < 1} decreases it.}
#'   \item{plot}{(only when \code{plot = TRUE}) a \pkg{ggplot2} object visualizing SAM over a grid.
#'     Illogical combinations of percentiles are shaded grey (e.g., \eqn{L_{2.5} > L_{97.5}}).}
#' }
#'
#' @details
#' The unconstrained multiplier is \eqn{(1 + lower\_adherence) \times (1 + upper\_adherence)}.
#' If any slot limit is violated, the multiplier is \code{pmin(constraint, multiplier)}.
#'
#' @examples
#' SAM(lower = 13, upper = 24, minLS = 12, maxLS = 24)                      # no violation
#' SAM(lower = 13, upper = 25, minLS = 12, maxLS = 24, constraint = 0.95)   # violation with constraint
#'
#' \donttest{
#' out <- SAM(
#'   lower = 13, upper = 25,
#'   minLS = 12, maxLS = 24,
#'   res = 0.5,
#'   lower_percentile = 5, upper_percentile = 95,
#'   constraint = 1,
#'   plot = TRUE,
#'   length_units = "cm")
#' out$SAM
#' }
#'
#' @seealso \code{\link{percentile}} for computing percentiles from length–frequency data.
#' @export
SAM <- function(lower = NULL, upper = NULL,
                minLS = NULL, maxLS = NULL,
                constraint = 1, digits = 2,
                plot = FALSE, res = 1,
                lower_percentile = 2.5, upper_percentile = 97.5,
                length_units = NULL) {

  # ---- validation ----
  nums <- list(lower = lower, upper = upper, minLS = minLS, maxLS = maxLS, constraint = constraint)
  bad_len <- vapply(nums, function(x) !is.numeric(x) || length(x) != 1L || !is.finite(x), logical(1))
  if (any(bad_len)) stop("All inputs must be finite, numeric scalars.", call. = FALSE)

  if (minLS <= 0 || maxLS <= 0 || lower < 0 || upper < 0)
    stop("All sizes must be non-negative and minLS,maxLS must be > 0.", call. = FALSE)

  if (minLS >= maxLS)
    stop("Require minLS < maxLS.", call. = FALSE)

  if (lower > upper)
    stop("Require lower <= upper.", call. = FALSE)

  if (constraint < 0 || constraint > 1)
    stop("`constraint` must be in '[0, 1]'.", call. = FALSE)

  if (!is.null(digits) && !is.na(digits)) {
    if (length(digits) != 1 || !is.numeric(digits) || digits < 0 || digits %% 1 != 0)
      stop("`digits` must be a single non-negative integer or NA.", call. = FALSE)
  }

  if (!is.logical(plot) || length(plot) != 1L || is.na(plot))
    stop("`plot` must be a single logical (TRUE/FALSE).", call. = FALSE)

  if (!is.numeric(res) || length(res) != 1L || !is.finite(res) || res <= 0)
    stop("`res` must be a single numeric > 0.", call. = FALSE)

  if (!is.null(length_units)) {
    if (!is.character(length_units) || length(length_units) != 1L ||
        is.na(length_units) || !nzchar(trimws(length_units))) {
      stop("`length_units` must be a non-empty character scalar or NULL.", call. = FALSE)
    }
    length_units <- trimws(length_units)
  }

  # --- label args validation ---
  if (!is.numeric(lower_percentile) || length(lower_percentile) != 1L || !is.finite(lower_percentile))
    stop("`lower_percentile` must be a single finite numeric.", call. = FALSE)
  if (!is.numeric(upper_percentile) || length(upper_percentile) != 1L || !is.finite(upper_percentile))
    stop("`upper_percentile` must be a single finite numeric.", call. = FALSE)

  # ---- adherence metrics ----
  lower_adherence <- (lower - minLS) / minLS
  upper_adherence <- (maxLS - upper) / maxLS

  # ---- multiplier ----
  multiplier <- (1 + lower_adherence) * (1 + upper_adherence)

  # apply constraint only if limits violated
  violated <- (lower < minLS) || (upper > maxLS)
  if (violated) multiplier <- min(constraint, multiplier)

  # ---- rounding policy ----
  if (!is.null(digits) && !is.na(digits)) {
    lower_out <- round(lower_adherence, digits)
    upper_out <- round(upper_adherence, digits)
    sam_out   <- round(multiplier, digits)
  } else {
    lower_out <- lower_adherence
    upper_out <- upper_adherence
    sam_out   <- multiplier
  }

  out <- list(
    lower_adherence = lower_out,
    upper_adherence = upper_out,
    SAM = sam_out
  )

  # ---- optional plot ----
  if (isTRUE(plot)) {
    if (!requireNamespace("ggplot2", quietly = TRUE))
      stop("Package 'ggplot2' is required for plotting.", call. = FALSE)

    .use_linewidth <- function() {
      requireNamespace("ggplot2", quietly = TRUE) &&
        utils::packageVersion("ggplot2") >= "3.4.0"
    }

    lower_range <- seq(0.1 * minLS, 1.9 * minLS, by = res)
    upper_range <- seq(0.1 * maxLS, 1.9 * maxLS, by = res)

    # Ensure the grid has >1 value in each axis (in case res is too large)
    if (length(lower_range) < 2L) lower_range <- unique(c(0.9 * minLS, minLS, 1.1 * minLS))
    if (length(upper_range) < 2L) upper_range <- unique(c(0.9 * maxLS, maxLS, 1.1 * maxLS))

    grid <- expand.grid(comp_lower = lower_range, comp_upper = upper_range, KEEP.OUT.ATTRS = FALSE)

    lower_ad <- (grid$comp_lower - minLS) / minLS
    upper_ad <- (maxLS - grid$comp_upper) / maxLS
    grid$SAM <- (1 + lower_ad) * (1 + upper_ad)

    violated_grid <- (grid$comp_lower < minLS) | (grid$comp_upper > maxLS)
    grid$SAM[violated_grid] <- pmin(constraint, grid$SAM[violated_grid])

    # Mark illogical percentile pairings as NA (to be drawn in grey via na.value)
    grid$SAM[grid$comp_lower > grid$comp_upper] <- NA_real_

    span <- suppressWarnings(max(abs(grid$SAM - 1), na.rm = TRUE))
    if (!is.finite(span) || span == 0) span <- 0.25
    lims <- c(1 - span, 1 + span)

    if (is.null(length_units)) {
      x_lab <- bquote(L[.(lower_percentile)])
      y_lab <- bquote(L[.(upper_percentile)])
    } else {
      x_lab <- bquote(L[.(lower_percentile)] ~ "(" * .(length_units) * ")")
      y_lab <- bquote(L[.(upper_percentile)] ~ "(" * .(length_units) * ")")
    }

    p <- ggplot2::ggplot(grid, ggplot2::aes(x = comp_lower, y = comp_upper, fill = SAM)) +
      ggplot2::geom_tile(alpha = 0.9, na.rm = FALSE) +
      ggplot2::scale_fill_gradient2(
        name = "",
        low = "darkred", mid = "lightyellow", high = "darkgreen",
        midpoint = 1, limits = lims, na.value = "grey80"
      ) +
      ggplot2::guides(
        fill = ggplot2::guide_colorbar(
          title = NULL,
          direction = "horizontal",
          barwidth = grid::unit(0.3, "npc"),
          barheight = grid::unit(10, "pt"),
          label.position = "bottom",
          ticks = FALSE
        )
      ) +
      ggplot2::geom_vline(
        xintercept = minLS, color = "black", linetype = "dashed",
        size = if (.use_linewidth()) NULL else 0.5,
        linewidth = if (.use_linewidth()) 0.5 else NULL
      ) +
      ggplot2::geom_hline(
        yintercept = maxLS, color = "black", linetype = "dashed",
        size = if (.use_linewidth()) NULL else 0.5,
        linewidth = if (.use_linewidth()) 0.5 else NULL
      ) +
      ggplot2::geom_point(
        x = lower, y = upper,
        color = "black", size = 2.5, inherit.aes = FALSE
      ) +
      ggplot2::labs(
        title = "SAM",
        x = x_lab, y = y_lab
      ) +
      ggplot2::coord_cartesian(xlim = range(lower_range), ylim = range(upper_range), expand = FALSE) +
      ggplot2::theme_bw() +
      ggplot2::theme(
        plot.title = ggplot2::element_text(hjust = 0.5),
        plot.margin = grid::unit(c(0.25, 0.25, 0.25, 0.25), "cm"),
        text = ggplot2::element_text(size = 15),
        legend.position = "top",
        legend.direction = "horizontal",
        legend.background = ggplot2::element_blank(),
        legend.key = ggplot2::element_blank()
      )

    out$plot <- p
  }

  out
}
