% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/convert_integer64.R
\name{convert_integer64}
\alias{convert_integer64}
\title{Convert \code{integer64} data to base R numeric, integer, or character types}
\usage{
convert_integer64(
  df,
  to_integer = "if_fits",
  precision_loss = "character",
  always_character = FALSE
)
}
\arguments{
\item{df}{A \code{data.frame}, list, or vector to process.}

\item{to_integer}{Character string controlling how conversion to integer is handled.
The rule is applied \strong{variable by variable}. Must be one of:
\itemize{
\item \code{"never"} — always convert to \code{numeric}, never to \code{integer}
\item \code{"if_fits"} — convert to \code{integer} if all values fit within 32-bit range \emph{(default)}
\item \code{"if_summable"} — convert to \code{integer} if the sum of absolute values fits within 32-bit range
\item \code{"always"} — always convert to \code{integer}, with potential coercion warnings
\item \code{"always_quiet"} — always convert to \code{integer}, suppressing coercion warnings
}}

\item{precision_loss}{Character string controlling what happens when
64-bit integers cannot be represented exactly as 64-bit floating-point numbers.
Must be one of:
\itemize{
\item \code{"character"} — convert to \code{character} if the value cannot be represented exactly \emph{(default)}
\item \code{"warn"} — convert to \code{numeric} and allow warnings about precision loss
\item \code{"quiet"} — convert to \code{numeric} but suppress such warnings
}}

\item{always_character}{Logical. If \code{TRUE}, all \code{integer64} values are converted
directly to \code{character}, overriding both \code{to_integer} and \code{precision_loss}.
Default is \code{FALSE}.}
}
\value{
The same type of object as the input (\code{data.frame}, list, or vector),
with all \code{integer64} values converted to base R \code{integer}, \code{numeric}, or
\code{character} depending on settings.
}
\description{
Converts \code{integer64} data (from the \strong{bit64} package) in a \code{data.frame},
list, or vector to base R numeric, integer, or character types.
}
\details{
Variables with class \code{integer64} often appear when reading data from Arrow
files, for example using \code{arrow::read_parquet()}. Arrow supports 64-bit
integer values, while the R language (and thus all R packages, including
the tidyverse) only supports 32-bit integers and 64-bit floating-point
numbers. These 64-bit integers therefore need conversion when loaded into R.

When the input is a \code{data.frame} or list, conversion is performed \strong{variable by variable},
and only those with class \code{integer64} are modified.

Depending on settings, \code{integer64} data are converted to base R \code{integer},
\code{numeric} (double), or \code{character}.

Note that a simpler helper that always converts directly to \code{numeric},
without any checks or dependency tests, can be defined as:

\if{html}{\out{<div class="sourceCode">}}\preformatted{              convert_integer64_to_numeric <- function(df) \{
                df[] <- lapply(df, function(x) \{
                  if (inherits(x, "integer64")) as.numeric(x) else x
                \})
                df
              \}
}\if{html}{\out{</div>}}
}
\note{
This function is written and documented with help from ChatGPT.
}
\examples{
if (requireNamespace("bit64", quietly = TRUE)) {
  x  <- bit64::seq.integer64(2025, 10^9, 3 * 10^8)
  print(x)
  
  print(convert_integer64(x*4, "always_quiet"))
  
  df <- data.frame(a = 11:14, b = x, c = 2 * x, d = 3 * x, e = x * x, f = c(22, 23, 24, 25))
  print(df)

  df1 <- convert_integer64(df, "never")
  df2 <- convert_integer64(df, "if_fits")
  df3 <- convert_integer64(df, "if_summable")
  df4 <- convert_integer64(df, "always_quiet")

  print(sapply(df,  class))
  print(sapply(df1, class))
  print(sapply(df2, class))
  print(sapply(df3, class))
  print(sapply(df4, class))
  
  print(df2)
  print(df4)
  
  cat("# Examples showing that integer64 is problematic:\n")
      y <- bit64::seq.integer64(1, 3)
      print(y)
      print(0.5 * y)
      print(y * 0.5)
      matrix(y, 1, 3)   
}

}
\seealso{
\code{\link[bit64:as.integer64.character]{bit64::as.integer64()}}
}
