#' Computation of the Kimura length
#'
#' \code{kimuraLength} - Computes the Kimura length according to
#' the specified formula to estimate the root length
#' shown in a skeletonized black & white image of a root. \cr
#' Optionally, a masking layer can be specified that indicates which pixels of
#' the image should be considered.
#'
#' @param skel_img Skeleton image (provided by the RootDetector). Can be a PNG,
#' i.e., an array with 3 dimensions (3 layers each containing a 2-dim. numeric
#' matrix with values between 0 and 1), or a 2-dim. matrix.
#' @param formula Integer value specifying the formula to be used (by default
#' 6) based on Kimura et al., 1999 (see references). Available are the
#' following formulas, where \eqn{N_o}{N_o} denotes the number of orthogonal
#' and \eqn{N_d}{N_d} the number of diagonal connections (between white pixels),
#' and \eqn{N_{whitepx}}{N_{whitepx}} the number of white pixels in the image.
#' Furthermore, there are three additional formulas to simply count white,
#' or black, or red pixels. \cr
#' - Formula 1: \eqn{1.1107 \cdot (N_d + N_o)}{1.1107 \* (N_d + N_o)},\cr
#' - Formula 2: \eqn{\sqrt{2} \cdot N_d + N_o}{sqrt(2) \* N_d + N_o},\cr
#' - Formula 3: \eqn{0.948 \cdot (\sqrt{2} \cdot N_d + N_o)}{0.948 \* (sqrt(2) \* N_d + N_o)},\cr
#' - Formula 4: \eqn{\sqrt{N_d^2+ (N_d+N_o)^2}}{sqrt(N_d^2+ (N_d+N_o)^2)},\cr
#' - Formula 6: \eqn{\sqrt{N_d^2+ (N_d+N_o/2)^2} + N_o/2}{sqrt(N_d^2+ (N_d+N_o/2)^2) + N_o/2},\cr
#' - Formula 7: \eqn{1.1107 \cdot N_{whitepx}}{1.1107 \* N_{whitepx}}.
#' - Formula 97: \eqn{N_{whitepx}}.
#' - Formula 98: \eqn{N_{blackpx}}.
#' - Formula 99: \eqn{N_{redpx}}.
#' @param mask 2-dim. true/false-matrix with the same number of rows and columns
#' as \code{skel_img} (optional, default = NULL, interpreted as a matrix
#' consisting only of TRUEs, i.e., nothing is "removed" from the image).
#' @param strict_mask Specifies how strictly the mask should be applied.
#' Available are:\cr
#' - TRUE (default): Connections between TRUE-pixels and neighboring
#' FALSE-pixels are not counted. As a result, the root length is likely to be
#' underestimated, especially if there are many TRUE-FALSE transitions in the
#' mask. \cr
#' - FALSE: Connections between TRUE-pixels and neighboring
#' FALSE-pixels are counted. As a result, the root length is likely to be
#' overestimated, especially if there are many TRUE-FALSE transitions in the
#' mask.
#' @param show_messages Specify if messages about the counts of orthogonal and
#' diagonal connection counts should be depicted (default TRUE).
#'
#' @return \code{kimuraLength} Numeric value (root length estimation).
#'
#' @references Kimura, K., Kikuchi, S. & Yamasaki, Si. Accurate root length measurement by image analysis. Plant and Soil 216, 117–127 (1999). doi: 10.1023/A:1004778925316
#'
#' @export
#' @rdname kimuraLength
#'
#' @examples
#' # This is a simple image with 2 diagonal and 1 orthogonal connections.
#' # With Formula 6 (default):
#' kimuraLength(matrix(c(
#'   1, 0, 0, 0,
#'   0, 1, 0, 0,
#'   0, 0, 1, 1
#' ), ncol = 4, nrow = 3, byrow = TRUE))
#' # With Formula 4:
#' kimuraLength(
#'   matrix(c(
#'     1, 0, 0, 0,
#'     0, 1, 0, 0,
#'     0, 0, 1, 1
#'   ), ncol = 4, nrow = 3, byrow = TRUE),
#'   formula = 4
#' )
#' # With Formula 6 and a mask which makes the function ignore the right side
#' # of the image. If stict_mask = TRUE, only 1 diagonal connection can be
#' # found. If set to FALSE, i.e., relaxed mask borders, then 2 diagonal
#' # connections are counted.
#' kimuraLength(
#'   skel_img = matrix(c(
#'     1, 0, 0, 0,
#'     0, 1, 0, 0,
#'     0, 0, 1, 1
#'   ), ncol = 4, nrow = 3, byrow = TRUE),
#'   mask = matrix(
#'     c(
#'       TRUE, TRUE, FALSE, FALSE,
#'       TRUE, TRUE, FALSE, FALSE,
#'       TRUE, TRUE, FALSE, FALSE
#'     ),
#'     ncol = 4, nrow = 3,
#'     byrow = TRUE
#'   ), strict_mask = FALSE
#' )
kimuraLength <- function(skel_img, formula = 6,
                         mask = NULL, strict_mask = TRUE,
                         show_messages = TRUE) {
  # If the image is only a matrix, give it three layers.
  if (length(dim(skel_img)) == 2) {
    skel_img <- array(c(skel_img, skel_img, skel_img),
      dim = c(dim(skel_img), 3)
    )
  }
  # If there is no mask specified, set it to TRUE:
  if (is.null(mask)) {
    mask <- matrix(TRUE, nrow = dim(skel_img)[1], ncol = dim(skel_img)[2])
  }

  # Some formulas do not need the counts of different connections:
  if (formula == 7) {
    num_white_px <- sum(apply(skel_img, MARGIN = c(1, 2), sum)[mask] == 3)
    kimura_length_px <- 1.1107 * num_white_px
    return(kimura_length_px)
  } else if (formula == 97) {
    num_white_px <- sum(skel_img[, , 1][mask] == 1 &
      skel_img[, , 2][mask] == 1 &
      skel_img[, , 3][mask] == 1)
    return(num_white_px)
  } else if (formula == 98) {
    num_black_px <- sum(skel_img[, , 1][mask] == 0 &
      skel_img[, , 2][mask] == 0 &
      skel_img[, , 3][mask] == 0)
    return(num_black_px)
  } else if (formula == 99) {
    num_red_px <- sum(skel_img[, , 1][mask] == 1 &
      skel_img[, , 2][mask] == 0 &
      skel_img[, , 3][mask] == 0)
    return(num_red_px)
  }

  # Get the image dimensions
  n_rows <- nrow(skel_img[, , 1])
  n_cols <- ncol(skel_img[, , 1])

  # Initialize counters for direct and diagonal connections
  orth_count <- 0
  diag_count <- 0

  # Define the direct and diagonal neighbor relative positions
  direct_neighbors <- list(c(0, 1), c(1, 0)) # Right, Down
  diagonal_neighbors <- list(c(1, 1), c(1, -1)) # Down-right, Down-left

  # Iterate over each pixel in the skeletonized image
  for (x in 1:n_rows) {
    for (y in 1:n_cols) {
      if (!mask[x, y]) {
        next
      }
      if (sum(skel_img[x, y, ] == 1) == 3) { # If the pixel is white (part of the skeleton)
        # Check for direct neighbors (right and down)
        for (neighbor in direct_neighbors) {
          nx <- x + neighbor[1]
          ny <- y + neighbor[2]
          if (strict_mask) {
            if (nx <= n_rows && ny <= n_cols &&
              mask[nx, ny] && # Difference: Neighbor must be in mask.
              sum(skel_img[nx, ny, ] == 1) == 3) {
              orth_count <- orth_count + 1
            }
          } else {
            if (nx <= n_rows && ny <= n_cols &&
              sum(skel_img[nx, ny, ] == 1) == 3) {
              orth_count <- orth_count + 1
            }
          }
        }
        # Check for diagonal neighbors (down-right and down-left)
        for (neighbor in diagonal_neighbors) {
          nx <- x + neighbor[1]
          ny <- y + neighbor[2]
          if (strict_mask) {
            if (nx <= n_rows && nx >= 1 &&
              ny <= n_cols && ny >= 1 &&
              mask[nx, ny] && # Difference: Neighbor must be in mask.
              sum(skel_img[nx, ny, ] == 1) == 3) {
              diag_count <- diag_count + 1
            }
          } else {
            if (nx <= n_rows && nx >= 1 &&
              ny <= n_cols && ny >= 1 &&
              sum(skel_img[nx, ny, ] == 1) == 3) {
              diag_count <- diag_count + 1
            }
          }
        }
      }
    }
  }
  if (show_messages) {
    message(paste(
      "Countet", diag_count, "diagonal connection(s) and", orth_count,
      "orthogonal connection(s)."
    ))
  }
  # Calculate the Kimura length
  kimura_length_px <- NULL
  if (formula == 1) {
    kimura_length_px <- 1.1107 * (diag_count + orth_count)
  } else if (formula == 2) {
    kimura_length_px <- sqrt(2) * diag_count + orth_count
  } else if (formula == 3) {
    kimura_length_px <- 0.948 * (sqrt(2) * diag_count + orth_count)
  } else if (formula == 4) {
    kimura_length_px <- sqrt(diag_count^2 + (diag_count + orth_count)^2)
  } else if (formula == 6) {
    kimura_length_px <- sqrt(diag_count^2 + (diag_count + orth_count / 2)^2) +
      orth_count / 2
  } else {
    stop("The specified formula is not available.")
  }
  return(kimura_length_px)
}
