% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/SegOne_v4_1_2.R
\name{SegOne}
\alias{SegOne}
\title{Single Tree Wood-Leaf Segmentation and Comprehensive Metrics Calculation}
\usage{
SegOne(a, filename = "Elab_single_tree", dimVox = 2, th = 2, 
       eps = 2, mpts = 6, N = 1000, R = 30, output_path = tempdir(),
       calculate_metrics = TRUE, voxel_size_canopy = 0.1, 
       coverage_method = "linear")
}
\arguments{
\item{a}{Input point cloud data. Can be either: (1) a data frame with x, y, z coordinates,
or (2) a file path to a .txt or .xyz file containing point cloud data}

\item{filename}{Character string specifying the output file prefix (default: "Elab_single_tree")}

\item{dimVox}{Numeric value specifying voxel dimension in centimeters for wood 
segmentation. Typical range: 1-5 cm. Smaller values increase computational cost 
but improve spatial resolution (default: 2)}

\item{th}{Integer specifying minimum number of points required to generate a voxel. 
Used to filter noise and low-density regions (default: 2)}

\item{eps}{Numeric value specifying the epsilon neighborhood radius for DBSCAN 
clustering in voxel units. Determines spatial connectivity of wood clusters (default: 2)}

\item{mpts}{Integer specifying minimum number of points required in epsilon neighborhood 
for a voxel to be considered a core point in DBSCAN algorithm (default: 6)}

\item{N}{Integer specifying minimum number of voxels required to form a valid wood 
cluster. Filters small non-wood clusters (default: 1000)}

\item{R}{Numeric threshold for cluster shape parameter (standard deviation proportion 
of variance from PCA). Used to identify cylindrical/linear wood structures. 
Higher values are more restrictive (default: 30)}

\item{output_path}{Character string specifying directory path for output files. 
If directory does not exist, it will be created (default: tempdir())}

\item{calculate_metrics}{Logical flag indicating whether to calculate comprehensive 
tree metrics. If FALSE, only wood-leaf segmentation is performed (default: TRUE)}

\item{voxel_size_canopy}{Numeric value specifying voxel size in meters for canopy 
volume calculation. Typical range: 0.05-0.2 m (default: 0.1)}

\item{coverage_method}{Character string specifying method for calculating coverage 
degree in canopy volume analysis. Options: "linear", "mean_normalized", 
"exponential", "threshold", "mediterranean" (default: "linear")}
}
\value{
Invisibly returns a named list containing:
\describe{
  \item{wood_file}{Character string with full path to wood component point cloud file}
  \item{leaf_file}{Character string with full path to foliage point cloud file}
  \item{metrics_file}{Character string with full path to metrics CSV file 
    (NULL if calculate_metrics = FALSE)}
  \item{metrics}{data.table containing calculated tree structural metrics 
    (NULL if calculate_metrics = FALSE)}
}
}
\description{
Performs wood-leaf segmentation and calculates comprehensive structural metrics for 
individual trees from terrestrial laser scanning (TLS) point cloud data. This function 
implements a unified approach consistent with the Forest_seg pipeline, ensuring 
methodological coherence across the PiC package.


The analysis follows a four-stage processing pipeline:
\enumerate{
  \item Voxelization and wood component identification using DBSCAN clustering
  \item Foliage separation through voxel-based subtraction
  \item Tree structural metrics calculation (height, DBH, crown base)
  \item Canopy volume quantification using density-weighted voxel analysis
}
}
\details{
## Processing Pipeline

**Stage 1: Wood Segmentation**

Wood components are identified through a multi-step process:
\enumerate{
  \item Point cloud voxelization at resolution specified by \code{dimVox}
  \item DBSCAN clustering applied to voxel centroids using \code{eps} and \code{mpts}
  \item Principal Component Analysis (PCA) filtering to identify cylindrical structures
  \item Retention of clusters with shape parameter R exceeding threshold (cylindrical wood)
}

The PCA-based filtering exploits the geometric properties of tree stems and branches,
which exhibit high first principal component values due to their elongated structure.

**Stage 2: Foliage Separation**

Foliage points are extracted using voxel-based subtraction:
\enumerate{
  \item Both wood and total point cloud are voxelized at 0.2 m resolution
  \item Wood-occupied voxels are identified
  \item Non-wood voxels are retained and mapped back to original points
}

This approach ensures complete spatial separation between wood and foliage components.

**Stage 3: Structural Metrics Calculation**

When \code{calculate_metrics = TRUE}, the following metrics are computed:

\itemize{
  \item **Tree Base Location (X, Y, Z_min)**: Coordinates of lowest wood point
  \item **Tree Height**: Vertical distance from base to highest point within 1 m buffer
  \item **DBH (Diameter at Breast Height)**: Calculated at 1.3 m using Pratt circle 
    fitting algorithm with +/- 5 cm tolerance. Valid range: 5-300 cm. DBH value is 
    always reported in CSV output. DBH_RMSE_cm column provides fit quality (max 5 cm 
    for validation). DBH_valido flag indicates whether measurement meets quality standards.
  \item **Crown Base Height**: Detected using vertical density analysis to filter 
    noise, followed by gap analysis. Uses 0.5 m bins with minimum 3 points per bin.
    Valid range: 0.5 m to 90% of tree height
}

**Stage 4: Canopy Volume Quantification**

Canopy volume metrics are calculated using density-weighted voxel analysis:

\enumerate{
  \item Foliage points voxelized at resolution \code{voxel_size_canopy}
  \item Point density calculated per voxel
  \item Coverage degree computed using specified \code{coverage_method}
  \item Two volume metrics calculated:
    \itemize{
      \item **Canopy Volume**: Total occupied voxel volume (m^3)
      \item **Occupied Volume**: Density-weighted volume accounting for point distribution
    }
  \item Coverage area computed from ground projection of occupied voxels (mq)
}

## Improvements in Version 2.0

**Enhanced Crown Base Calculation:**
\itemize{
  \item Vertical density analysis filters noise points (isolated points near trunk)
  \item Uses 0.5 m vertical bins with minimum 3 points per bin threshold
  \item Combines density filtering with gap analysis for robust detection
  \item Validates results with multiple criteria
}
**Improved DBH Validation:**
\itemize{
  \item Updated maximum diameter to 3.0 m (300 cm) for large/monumental trees
  \item RMSE quality check (max 5 cm) used to flag poor circle fits
  \item DBH value always reported (even if RMSE threshold exceeded)
  \item DBH_RMSE_cm column provides fit quality indicator
  \item DBH_valido flag indicates whether measurement meets all quality standards
  \item Enhanced diagnostic messages showing fit quality
}
## Coverage Degree Methods

The \code{coverage_method} parameter determines how point density is translated 
to coverage degree:

\itemize{
  \item **linear**: Linear normalization by column maximum: \eqn{CD = N / N_{max}}
  \item **mean_normalized**: Normalization by mean density: \eqn{CD = N / \bar{N}}
  \item **exponential**: Exponential saturation: \eqn{CD = 1 - exp(-N / \bar{N})}
  \item **threshold**: Binary classification at 50th percentile
  \item **mediterranean**: Power-law scaling optimized for sparse Mediterranean canopies: 
    \eqn{CD = (N / N_{max})^{0.7}}
}

For single trees, "linear" is recommended as it provides intuitive interpretation 
of point density relative to maximum observed density.

## Quality Assurance

The function implements several validation checks:

\itemize{
  \item DBH validation: radius 2.5-150 cm, minimum 5 points, RMSE reported (< 5 cm for valid flag)
  \item Crown base validation: density filtering, minimum 0.5 m, maximum 90% of height
  \item Point count validation: sufficient points in measurement zones
  \item Comprehensive error handling with diagnostic messages
}

## Output Files

Three files are generated in \code{output_path}:

\enumerate{
  \item **Wood points**: \code{<filename>_Wood_eps<eps>_mpts<mpts>.txt}
    \itemize{
      \item Format: x, y, z, cls (cluster ID)
      \item Contains all points classified as wood components
    }
  \item **Foliage points**: \code{<filename>_AGBnoWOOD_eps<eps>_mpts<mpts>.txt}
    \itemize{
      \item Format: x, y, z
      \item Contains all non-wood vegetation points
    }
  \item **Metrics**: \code{<filename>_metrics.csv} (if calculate_metrics = TRUE)
    \itemize{
      \item Contains all calculated structural metrics
      \item DBH_cm: Always populated when calculation succeeds
      \item DBH_RMSE_cm: Fit quality indicator (lower is better, <5 cm is valid)
      \item Semicolon-delimited format
    }
}
}
\note{
Version 2.0 addresses two critical issues identified in field testing:
\enumerate{
  \item Crown base calculation now robust against noise points near trunk
  \item DBH validation extended to 3 m diameter with quality checks
}

This implementation is fully consistent with the Forest_seg approach, ensuring 
methodological coherence across the PiC package.
}
\section{Parameter Selection Guidelines}{


**Voxel Size (dimVox)**:
\itemize{
  \item Small trees or fine branches: 1-2 cm
  \item Medium trees: 2-3 cm
  \item Large trees: 3-5 cm
}

**DBSCAN Parameters (eps, mpts)**:
\itemize{
  \item Densely scanned trees: eps = 1-2, mpts = 4-6
  \item Sparsely scanned trees: eps = 2-3, mpts = 3-4
  \item Complex branch structures: lower eps, higher mpts
}

**Cluster Size (N)**:
\itemize{
  \item Small trees: N = 500-1000
  \item Medium trees: N = 1000-2000
  \item Large trees: N = 2000-5000
}
}

\examples{
\dontrun{
# Basic usage with default parameters
result <- SegOne(
  a = "single_tree.xyz",
  filename = "tree_analysis",
  output_path = "~/results"
)

# View calculated metrics
print(result$metrics)

# Advanced usage for large tree
result <- SegOne(
  a = my_point_cloud,
  filename = "large_oak",
  dimVox = 3,           # Larger voxels for large tree
  eps = 2,              # Increased connectivity
  mpts = 6,             # More stringent clustering
  N = 2000,             # Larger minimum cluster size
  R = 35,               # Stricter cylindrical filter
  output_path = "~/tree_metrics",
  voxel_size_canopy = 0.15,
  coverage_method = "linear"
)
}

}
\references{
Ferrara, R., Virdis, S.G.P., Ventura, A., Ghisu, T., Duce, P., & Pellizzaro, G. (2018).
An automated approach for wood-leaf separation from terrestrial LIDAR point clouds 
using the density based clustering algorithm DBSCAN. Agricultural and Forest Meteorology, 
262, 434-444. \doi{10.1016/j.agrformet.2018.04.008}

Pratt, V. (1987). Direct least-squares fitting of algebraic surfaces. 
ACM SIGGRAPH Computer Graphics, 21(4), 145-152.
}
