#' @title Stress Tolerance Index (STI)
#' @description
#' This function computes stress tolerance index (STI) using any traits (like yield) under stress and non-stress conditions. The lower values of STI indicates greater tolerance. For more details see Fernandez (1992).
#' @param Gen Character vector of genotype identifiers.
#' @param YN Numeric vector: yield (any trait) under non-stress (normal) environment.
#' @param YS Numeric vector: yield (any trait) under stress environment.
#' @param YMN Optional numeric scalar: mean of \code{YN}. If NULL, computed as \code{mean(YN, na.rm=TRUE)}.
#' @return A list of components:
#' \itemize{
#' \item \code{YMN}: Mean of yield (any trait) values under normal condition
#' \item \code{Result}: It includes
#' \itemize{
#' \item \code{Gen}: Character vector of genotype IDs.
#' \item \code{STI}: Estimated stress tolerance index (STI) values for respective genotypes.
#' }
#'}
#' \deqn{STI = \frac{YN \times YS}{(YMN)^2}}
#'
#' @references Fernandez, G.C.J. (1992). Effective selection criteria for assessing plant stress tolerance. In: Proceedings of the international symposium on adaptation of vegetables and other food crops in temperature and water stress. AVRDC Publication: Tainan, Taiwan: Shanhua: Chapter (25), 257–270.
#' @examples
#' out=STI(Gen=c("G1","G2","G3"), YN=c(10,8,5), YS=c(7,5,3))
#' print(out)
#' @export
STI <- function(Gen, YN, YS, YMN = NULL) {
  .check_lengths(Gen, YN, YS)
  YN <- as.numeric(YN); YS <- as.numeric(YS)
  if (is.null(YMN)) YMN <- mean(YN, na.rm = TRUE)
  denom <- (YMN^2)
  out <- (YN * YS) / denom
  out[denom == 0] <- NA_real_
  Result <- data.frame(Gen = as.character(Gen), STI = out, row.names = NULL)
  return(list(YMN = YMN, Result = Result))
}
