% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ccsr_map.R
\name{ccsr_map}
\alias{ccsr_map}
\title{Map ICD-10 Codes to CCSR Categories}
\usage{
ccsr_map(
  data,
  code_col,
  map_df,
  type = NULL,
  default_only = FALSE,
  output_format = "long",
  keep_all = TRUE
)
}
\arguments{
\item{data}{A data frame or tibble containing ICD-10 codes to be mapped.}

\item{code_col}{Character string specifying the name of the column in \code{data}
that contains the ICD-10 codes.}

\item{map_df}{A tibble containing the CCSR mapping data, typically obtained
from \code{download_ccsr()}.}

\item{type}{Character string specifying the type of mapping. Must be one of:
"diagnosis" (or "dx") for ICD-10-CM codes, or "procedure" (or "pr") for
ICD-10-PCS codes. If NULL (default), the function will attempt to infer
the type from the mapping data frame.}

\item{default_only}{Logical. For diagnosis codes only, if TRUE, returns only
the default CCSR category (recommended for principal diagnosis analysis).
If FALSE (default), returns all assigned CCSR categories including
cross-classifications.}

\item{output_format}{Character string specifying the output format. Must be
one of: "long" (default) or "wide". "long" format duplicates records for
each assigned CCSR category. "wide" format creates multiple columns
(CCSR_1, CCSR_2, etc.) for multiple categories.}

\item{keep_all}{Logical. If TRUE (default), returns all original columns
from \code{data} plus the CCSR mapping columns. If FALSE, returns only the
ICD-10 code column and CCSR mapping columns.}
}
\value{
A tibble with the original data plus CCSR mapping columns. The
structure depends on \code{output_format}:
\itemize{
\item For "long" format: Each row represents one ICD-10 code and one CCSR
category assignment (rows are duplicated for multiple categories).
\item For "wide" format: Each row represents one ICD-10 code with multiple
CCSR category columns (CCSR_1, CCSR_2, etc.).
}
}
\description{
Maps ICD-10-CM diagnosis codes or ICD-10-PCS procedure codes to their
corresponding CCSR categories using a downloaded CCSR mapping file.
}
\details{
CCSR allows for cross-classification, meaning a single ICD-10 code can map
to multiple CCSR categories. The "long" format is recommended for analyses
where you want to count all assigned CCSR categories, while "wide" format
may be more convenient for patient-level analyses.

For diagnosis codes, CCSR also assigns a "default" category that is
recommended for principal diagnosis analysis. Use \code{default_only = TRUE} to
extract only this default category.
}
\examples{
\donttest{
# Download mapping file
dx_map <- download_ccsr("diagnosis")

# Create sample data
sample_data <- tibble::tibble(
  patient_id = 1:3,
  icd10_code = c("E11.9", "I10", "M79.3")
)

# Map codes (long format - default)
mapped_long <- ccsr_map(
  data = sample_data,
  code_col = "icd10_code",
  map_df = dx_map
)

# Map codes (wide format)
mapped_wide <- ccsr_map(
  data = sample_data,
  code_col = "icd10_code",
  map_df = dx_map,
  output_format = "wide"
)

# Map codes (default category only)
mapped_default <- ccsr_map(
  data = sample_data,
  code_col = "icd10_code",
  map_df = dx_map,
  default_only = TRUE
)
}

}
