% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/easyEGO.cst.R
\name{easyEGO.cst}
\alias{easyEGO.cst}
\title{EGO algorithm with constraints}
\usage{
easyEGO.cst(
  fun,
  constraint,
  n.cst = 1,
  budget,
  lower,
  upper,
  cheapfun = FALSE,
  equality = FALSE,
  X = NULL,
  y = NULL,
  C = NULL,
  control = list(method = "EFI", trace = 1, inneroptim = "genoud", maxit = 100, seed =
    42),
  ...
)
}
\arguments{
\item{fun}{scalar function to be minimized,}

\item{constraint}{vectorial function corresponding to the constraints, see details below,}

\item{n.cst}{number of constraints,}

\item{budget}{total number of calls to the objective and constraint functions,}

\item{lower}{vector of lower bounds for the variables to be optimized over,}

\item{upper}{vector of upper bounds for the variables to be optimized over,}

\item{cheapfun}{optional boolean, \code{TRUE} if the objective is a fast-to-evaluate function that does not need a kriging model}

\item{equality}{either \code{FALSE} if all constraints are inequalities, else a Boolean vector indicating which are equalities}

\item{X}{initial design of experiments. If not provided, X is taken as a maximin LHD with budget/3 points}

\item{y}{initial set of objective observations \eqn{f(X)}. Computed if not provided.}

\item{C}{initial set of constraint observations \eqn{g(X)}. Computed if not provided.}

\item{control}{an optional list of control parameters. See "Details".}

\item{...}{additional parameters to be given to BOTH the objective \code{fun} and \code{constraints}.}
}
\value{
A list with components:
\itemize{
\item \code{par}: the best feasible point
\item \code{values}: a vector of the objective and constraints at the point given in \code{par},
\item \code{history}: a list containing all the points visited by the algorithm (\code{X}) and their corresponding objectives (\code{y}) and constraints (\code{C}) 
\cr \cr
If no feasible point is found, \code{par} returns the most feasible point (in the least square sense).
}
}
\description{
User-friendly wrapper of the function \code{\link[DiceOptim]{EGO.cst}}
Generates initial DOEs and kriging models (objects of class \code{\link[DiceKriging]{km}}), 
and executes \code{nsteps} iterations of EGO methods integrating constraints.
}
\details{
Does not require knowledge on kriging models (objects of class \code{\link[DiceKriging]{km}})  \cr

The problem considered is of the form: \eqn{min f(x)} s.t. \eqn{g(x) \le 0}, 
\eqn{g} having a vectorial output. 
By default all its components are supposed to be inequalities, but one can use a Boolean vector in \code{equality} 
to specify which are equality constraints, hence of the type \eqn{g(x) = 0}.
The \code{control} argument is a list that can supply any of the following components: \cr
\itemize{
\item \code{method}: choice of constrained improvement function: "\code{AL}", "\code{EFI}" or "\code{SUR}" 
(see \code{\link[DiceOptim]{crit_EFI}}, \code{\link[DiceOptim]{crit_AL}}, \code{\link[DiceOptim]{crit_SUR_cst}})
\item \code{trace}:  if positive, tracing information on the progress of the optimization is produced.
\item \code{inneroptim}: choice of the inner optimization algorithm: "\code{genoud}" or "\code{random}"
 (see \code{\link[rgenoud]{genoud}}).
\item \code{maxit}: maximum number of iterations of the inner loop.
\item \code{seed}: to fix the random variable generator
}
For additional details, see \code{\link[DiceOptim]{EGO.cst}}.
}
\examples{
#---------------------------------------------------------------------------
# 2D objective function, 3 cases
#---------------------------------------------------------------------------
\donttest{
set.seed(25468)
library(DiceDesign)

n_var <- 2 
fun <- goldsteinprice
fun1.cst <- function(x){return(-branin(x) + 25)}
fun2.cst <- function(x){return(3/2 - x[1] - 2*x[2] - .5*sin(2*pi*(x[1]^2 - 2*x[2])))}

# Constraint function with vectorial output
constraint <- function(x){return(c(fun1.cst(x), fun2.cst(x)))}

# For illustration purposes
n.grid <- 31
test.grid <- expand.grid(X1 = seq(0, 1, length.out = n.grid), X2 = seq(0, 1, length.out = n.grid))
obj.grid <- apply(test.grid, 1, fun)
cst1.grid <- apply(test.grid, 1, fun1.cst)
cst2.grid <- apply(test.grid, 1, fun2.cst)

lower <- rep(0, n_var)
upper <- rep(1, n_var)

#---------------------------------------------------------------------------
# 1- Expected Feasible Improvement criterion, expensive objective function,
# two inequality constraints, 15 observations budget, using genoud
#---------------------------------------------------------------------------
res <- easyEGO.cst(fun=fun, constraint=constraint, n.cst=2, lower=lower, upper=upper, budget=15, 
                   control=list(method="EFI", inneroptim="genoud", maxit=20))

cat("best design found:", res$par, "\n")
cat("corresponding objective and constraints:", res$value, "\n")

# Objective function in colour, constraint boundaries in red
# Initial DoE: white circles, added points: blue crosses, best solution: red cross

filled.contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), nlevels = 50,
               matrix(obj.grid, n.grid), main = "Two inequality constraints",
               xlab = expression(x[1]), ylab = expression(x[2]), color = terrain.colors, 
               plot.axes = {axis(1); axis(2);
                            contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), 
                                    matrix(cst1.grid, n.grid), level = 0, add=TRUE,
                                    drawlabels=FALSE, lwd=1.5, col = "red")
                            contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), 
                                    matrix(cst2.grid, n.grid), level = 0, add=TRUE,drawlabels=FALSE,
                                    lwd=1.5, col = "red")
                            points(res$history$X, col = "blue", pch = 4, lwd = 2)       
                            points(res$par[1], res$par[2], col = "red", pch = 4, lwd = 2, cex=2) 
               }
)

#---------------------------------------------------------------------------
# 2- Augmented Lagrangian Improvement criterion, expensive objective function,
# one inequality and one equality constraint, 25 observations budget, using random search
#---------------------------------------------------------------------------
res2 <- easyEGO.cst(fun=fun, constraint=constraint, n.cst=2, lower=lower, upper=upper, budget=25,
                   equality = c(TRUE, FALSE),
                   control=list(method="AL", inneroptim="random", maxit=100))

cat("best design found:", res2$par, "\n")
cat("corresponding objective and constraints:", res2$value, "\n")

# Objective function in colour, inequality constraint boundary in red, equality
# constraint in orange
# Initial DoE: white circles, added points: blue crosses, best solution: red cross

filled.contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), nlevels = 50,
               matrix(obj.grid, n.grid), xlab = expression(x[1]), ylab = expression(x[2]),
               main = "Inequality (red) and equality (orange) constraints", color = terrain.colors, 
               plot.axes = {axis(1); axis(2);
                            contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), 
                                    matrix(cst1.grid, n.grid), level = 0, add=TRUE,
                                    drawlabels=FALSE,lwd=1.5, col = "orange")
                            contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid),
                                    matrix(cst2.grid, n.grid), level = 0, add=TRUE,
                                    drawlabels=FALSE,lwd=1.5, col = "red")
                            points(res2$history$X, col = "blue", pch = 4, lwd = 2)
                            points(res2$par[1], res2$par[2], col = "red", pch = 4, lwd = 2, cex=2) 
               }
)

#---------------------------------------------------------------------------
# 3- Stepwise Uncertainty Reduction criterion, fast objective function,
# single inequality constraint, with initial DOE given + 10 observations,
# using genoud
#---------------------------------------------------------------------------
n.init <- 12 
design.grid <- round(maximinESE_LHS(lhsDesign(n.init, n_var, seed = 42)$design)$design, 1)
cst2.init <- apply(design.grid, 1, fun2.cst)

res3 <- easyEGO.cst(fun=fun, constraint=fun2.cst, n.cst=1, lower=lower, upper=upper, budget=10,
                    X=design.grid, C=cst2.init,
                   cheapfun=TRUE, control=list(method="SUR", inneroptim="genoud", maxit=20))
        
cat("best design found:", res3$par, "\n")
cat("corresponding objective and constraint:", res3$value, "\n")

# Objective function in colour, inequality constraint boundary in red
# Initial DoE: white circles, added points: blue crosses, best solution: red cross
                            
filled.contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), nlevels = 50,
               matrix(obj.grid, n.grid), main = "Single constraint, fast objective",
               xlab = expression(x[1]), ylab = expression(x[2]), color = terrain.colors, 
               plot.axes = {axis(1); axis(2);
                            points(design.grid[,1], design.grid[,2], pch = 21, bg = "white")
                            contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), 
                                    matrix(obj.grid, n.grid), nlevels = 10, add = TRUE,
                                    drawlabels = TRUE)
                            contour(seq(0, 1, length.out = n.grid), seq(0, 1, length.out = n.grid), 
                                    matrix(cst2.grid, n.grid), level = 0, add = TRUE,
                                    drawlabels = FALSE,lwd = 1.5, col = "red")
                            points(res3$history$X, col = "blue", pch = 4, lwd = 2)
                            points(res3$par[1], res3$par[2], col = "red", pch = 4, lwd = 2, cex=2) 
                                           }
                )
}

}
\references{
D.R. Jones, M. Schonlau, and W.J. Welch (1998), Efficient global
optimization of expensive black-box functions, \emph{Journal of Global
Optimization}, 13, 455-492.


M. Schonlau, W.J. Welch, and D.R. Jones (1998),
 Global versus local search in constrained optimization of computer models,
 \emph{Lecture Notes-Monograph Series}, 11-25.
 
M.J. Sasena, P. Papalambros, and P.Goovaerts (2002),
 Exploration of metamodeling sampling criteria for constrained global optimization,
 \emph{Engineering optimization}, 34, 263-278.
 
R.B. Gramacy, G.A. Gray, S. Le Digabel, H.K.H Lee, P. Ranjan, G. Wells, Garth, and S.M. Wild (2014+),
Modeling an augmented Lagrangian for improved blackbox constrained optimization,
\emph{arXiv preprint arXiv:1403.4890}.

J.M. Parr (2012),
\emph{Improvement criteria for constraint handling and multiobjective optimization},
University of Southampton.

V. Picheny (2014),
A stepwise uncertainty reduction approach to constrained global optimization,
\emph{Proceedings of the 17th International Conference on Artificial Intelligence and Statistics},  JMLR W&CP 33, 787-795.
}
\author{
Victor Picheny 

Mickael Binois
}
