% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/DPI.R
\name{DPI_dag}
\alias{DPI_dag}
\title{Directed acyclic graphs (DAGs) via DPI exploratory analysis (causal discovery) for all significant partial \emph{r}s.}
\usage{
DPI_dag(
  data,
  k.covs = 1,
  n.sim = 1000,
  alpha = 0.05,
  bonf = FALSE,
  pseudoBF = FALSE,
  seed = NULL,
  progress,
  file = NULL,
  width = 6,
  height = 4,
  dpi = 500
)
}
\arguments{
\item{data}{A dataset with at least 3 variables.}

\item{k.covs}{An integer vector (e.g., \code{1:10}) of number of random covariates (simulating potential omitted variables) added to each simulation sample. Defaults to \code{1}. For details, see \code{\link[=DPI]{DPI()}}.}

\item{n.sim}{Number of simulation samples. Defaults to \code{1000}.}

\item{alpha}{Significance level for computing the \code{Significance} score (0~1) based on \emph{p} value of partial correlation between \code{X} and \code{Y}. Defaults to \code{0.05}.
\itemize{
\item \code{Direction = R2.Y - R2.X}
\item \code{Significance = 1 - tanh(p.beta.xy/alpha/2)}
}}

\item{bonf}{Bonferroni correction to control for false positive rates: \code{alpha} is divided by, and \emph{p} values are multiplied by, the number of comparisons.
\itemize{
\item Defaults to \code{FALSE}: No correction, suitable if you plan to test only one pair of variables.
\item \code{TRUE}: Using \code{k * (k - 1) / 2} (all pairs of variables) where \code{k = length(data)}.
\item A user-specified number of comparisons.
}}

\item{pseudoBF}{Use normalized pseudo Bayes Factors \code{sigmoid(log(PseudoBF10))} alternatively as the \code{Significance} score (0~1). Pseudo Bayes Factors are computed from \emph{p} value of X-Y partial relationship and total sample size, using the transformation rules proposed by Wagenmakers (2022) \doi{10.31234/osf.io/egydq}.

Defaults to \code{FALSE} because it makes less penalties for insignificant partial relationships between \code{X} and \code{Y}, see Examples in \code{\link[=DPI]{DPI()}} and \href{https://psychbruce.github.io/DPI/#step-2-normalized-penalty-as-significance-score}{online documentation}.}

\item{seed}{Random seed for replicable results. Defaults to \code{NULL}.}

\item{progress}{Show progress bar. Defaults to \code{TRUE} (if \code{length(k.covs)} >= 5).}

\item{file}{File name of saved plot (\code{".png"} or \code{".pdf"}).}

\item{width, height}{Width and height (in inches) of saved plot. Defaults to \code{6} and \code{4}.}

\item{dpi}{Dots per inch (figure resolution). Defaults to \code{500}.}
}
\value{
Return a data.frame (class \code{dpi.dag}) of DPI exploration results.
}
\description{
Directed acyclic graphs (DAGs) via DPI exploratory analysis (causal discovery) for all significant partial \emph{r}s.
}
\examples{
\donttest{# partial correlation networks (undirected)
cor_net(airquality, "pcor")

# directed acyclic graphs (grey edge = insignificant DPI)
dpi.dag = DPI_dag(airquality, k.covs=c(1,3,5), seed=1)
print(dpi.dag, k=1)  # DAG with DPI(k=1)
print(dpi.dag, k=3)  # DAG with DPI(k=3)
print(dpi.dag, k=5)  # DAG with DPI(k=5)

# set edge labels and edge transparency
# (grey edge = insignificant DPI)
print(dpi.dag, k=5, show.label=FALSE, faded.dpi=TRUE)

# modify ggplot attributes
gg = plot(dpi.dag, k=5, show.label=FALSE, faded.dpi=TRUE)
gg + labs(title="DAG with DPI (k=5)")

# visualize DPIs of multiple paths
ggplot(dpi.dag$DPI, aes(x=k.cov, y=DPI)) +
  geom_ribbon(
    aes(ymin=Sim.LLCI, ymax=Sim.ULCI, fill=path),
        alpha=0.1) +
  geom_line(aes(color=path), linewidth=0.7) +
  geom_point(aes(color=path)) +
  geom_hline(yintercept=0, color="red",
             linetype="dashed") +
  scale_y_continuous(limits=c(NA, 0.5)) +
  labs(color="Directed Prediction",
       fill="Directed Prediction") +
  theme_classic()
}
}
\seealso{
\link{S3method.network}

\code{\link[=DPI]{DPI()}}

\code{\link[=DPI_curve]{DPI_curve()}}

\code{\link[=BNs_dag]{BNs_dag()}}

\code{\link[=cor_net]{cor_net()}}

\code{\link[=p_to_bf]{p_to_bf()}}
}
