% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ari.R
\name{ari}
\alias{ari}
\alias{print.ari}
\title{Computes the adjusted Rand index and the confidence interval, comparing
two classifications from a contingency table.}
\usage{
ari(mat, alpha = 0.05, digits = 2)

\method{print}{ari}(x, ...)
}
\arguments{
\item{mat}{A matrix of integers representing the contingency table of
reference}

\item{alpha}{A single number strictly included between 0 and 1
representing the significance level of interest. (default is 0.05)}

\item{digits}{An integer for the returned significant digits to return
(default is 2)}

\item{x}{an object used to select a method.}

\item{...}{further arguments passed to or from other methods.}
}
\value{
An object of class \code{ari} with the following elements:
\item{AdjustedRandIndex}{The adjusted Rand Index}
\item{CI}{The confidence interval}
}
\description{
Computes the adjusted Rand index and the confidence interval, comparing
two classifications from a contingency table.

print method for ari class
}
\details{
The adjusted Rand Index (ARI) should be interpreted as follows:

ARI >= 0.90 excellent recovery;
0.80 =< ARI < 0.90 good recovery;
0.65 =< ARI < 0.80 moderate recovery;
ARI < 0.65 poor recovery.

As the confidence interval is based on the approximation to the Normal
distribution, it is recommended to trust in the confidence interval only
in cases of total number of object clustered greater than 100.
}
\section{Methods (by generic)}{
\itemize{
\item \code{print(ari)}: 

}}
\examples{

#### This example compares the adjusted Rand Index as computed on the
### partitions given by Ward's algorithm with the ground truth on the
### famous Iris data set by the adjustedRandIndex function
### {mclust package} and by the ari function.

library(CrossClustering)
library(mclust)

clusters <- iris[-5] |>
  dist() |>
  hclust(method = 'ward.D') |>
  cutree(k = 3)

ground_truth <- iris[[5]] |> as.numeric()

mc_ari <- adjustedRandIndex(clusters, ground_truth)
mc_ari

ari_cc <- table(ground_truth, clusters) |>
  ari(digits = 7)
ari_cc

all.equal(mc_ari, unclass(ari_cc)[["ari"]], check.attributes = FALSE)

}
\references{
L. Hubert and P. Arabie (1985) Comparing partitions, Journal of
Classification, 2, 193-218.

E.M. Qannari, P. Courcoux and Faye P. (2014) Significance test of the
adjusted Rand index. Application to the free sorting task, Food Quality
and Preference, (32)93-97

M.H. Samuh, F. Leisch, and L. Finos (2014), Tests for Random Agreement
in Cluster Analysis, Statistica Applicata-Italian Journal of Applied
Statistics, vol. 26, no. 3, pp. 219-234.

D. Steinley (2004) Properties of the Hubert-Arabie Adjusted Rand Index,
Psychological Methods, 9(3), 386-396

D. Steinley, M.J. Brusco, L. Hubert (2016) The Variance of the Adjusted
Rand Index, Psychological Methods, 21(2), 261-272
}
\author{
Paola Tellaroli, <paola \code{dot} tellaroli \code{at} unipd \code{dot} it>;
}
