#' Extract contiguous sleep or wake periods from a labeled minute series
#'
#' Splits a minute-level, labeled time series into contiguous episodes of a
#' target state (sleep or wake) and returns the **activity vectors** for those
#' episodes whose length is at least `min_len` minutes.
#'
#' @param labeled_df A `data.frame` containing at least:
#'   \itemize{
#'     \item `Activity` — numeric minute-level activity.
#'     \item `label.sw` — binary sleep/wake label per minute (`1` = sleep,
#'           `0` = wake). `NA` values are treated as breaks between episodes.
#'   }
#' @param target_state Integer `0` or `1`. Use `0` to extract sleep episodes,
#'   `1` to extract wake episodes. Default is `1`.
#' @param min_len Integer minimum episode length **in minutes** (i.e., number of
#'   consecutive samples) required to keep an episode. Default is `30`.
#'
#' @return A **list** of numeric vectors. Each element is the `Activity` values
#' for one qualifying episode of the requested state. If no episode qualifies,
#' returns an empty list (`list()`).
#'
#' @details
#' The function uses run-length encoding over `label.sw` to identify contiguous
#' episodes. Any `NA` in `label.sw` is converted to a sentinel and treated as a
#' hard break (i.e., episodes do not cross `NA` gaps). Length filtering is
#' applied on the number of minutes (rows) per episode.
#'
#' @seealso   [sleep_detection()]
#'
#'#' @export

extract_sw_period <- function(labeled_df,  target_state = 1, min_len = 30) {
  sw  <- labeled_df$label.sw
  st  <- ifelse(is.na(sw), -9999, sw)          # treat NA as a break
  grp <- cumsum(c(1, st[-1] != st[-length(st)]))     # each period has a new label
  runs <- rle(st)
  keep <- (runs$values == as.integer(target_state)) &
    (runs$lengths >= as.integer(min_len))
  unname(split(labeled_df$Activity, grp)[keep])   # list of numeric vectors
}

