#' Nonparametric circadian metrics (RA, IS, IV, L5, M10)
#'
#' Computes five nonparametric metrics from minute-level activity:
#' Relative Amplitude (RA), Interdaily Stability (IS),
#' Intradaily Variability (IV), the least-active 5-hour block (L5),
#' and the most-active 10-hour block (M10).
#'
#' @param df cleaned data frame containing the activity varialbe at 1-minute epoch
#' @param L_window Integer window length (minutes) for L5 (default 300).
#' @param M_window Integer window length (minutes) for M10 (default 600).
#'
#' @return A list with components:
#' \describe{
#'   \item{RA}{Relative amplitude, \eqn{(M10 - L5) / (M10 + L5)} when denominator \eqn{> 0}.}
#'   \item{IS}{Interdaily stability (0..1).}
#'   \item{IV}{Intradaily variability (\eqn{\ge 0}).}
#'   \item{L5_mean}{Mean activity in the lowest 5-hour block of the 24 h profile.}
#'   \item{L5_start_min}{Minute-of-day (0~1439) at which L5 starts.}
#'   \item{M10_mean}{Mean activity in the highest 10-hour block of the 24 h profile.}
#'   \item{M10_start_min}{Minute-of-day (0~1439) at which M10 starts.}
#'   \item{profile_24h}{Length-1440 vector of minute-of-day means.}
#' }
#'
#' @seealso [screen_wear()], [sleep_detection()], [extract_sleep_metrics]

#' @export

extract_nonparametric_metrics <- function(df,
                                          L_window = 5 * 60,   # 5 hours
                                          M_window = 10 * 60)  # 10 hours
{
  n <- length(df$Activity)
  idx_first_midnight <- min(which(df$Time == "0:00:00" |df$Time == "00:00:00"))
  df = df[idx_first_midnight:n,]
  n <- length(df$Activity)

  x <- as.numeric(df$Activity)

  if (n < 3*1440) stop("Need at least three full day.")

  # Keep whole days only
  idx_first_midnight <- min(which(df$Time == "0:00:00"))

  ndays <- n %/% 1440 ## integer days
  x <- x[seq_len(ndays * 1440)]

  # Reshape to days × minutes (each row = a day)
  mat <- matrix(x, nrow = ndays, ncol = 1440, byrow = TRUE)

  # 24h mean profile (minute-of-day means)
  prof <- colMeans(mat, na.rm = TRUE)
  if (anyNA(prof)) {
    prof[is.na(prof)] <- mean(prof, na.rm = TRUE)  # simple fill to avoid NA in rolling means
  }

  # Circular rolling means for L5 and M10 (wrap profile once)
  roll_mean <- function(v, k) {
    v2 <- c(v, v)
    s  <- stats::filter(v2, rep(1, k), sides = 1)  # rolling sum
    as.numeric(s[k:(k + 1440 - 1L)] / k)
  }

  L5_series  <- roll_mean(prof, L_window)
  M10_series <- roll_mean(prof, M_window)

  L5_start   <- which.min(L5_series) - 1
  M10_start  <- which.max(M10_series) - 1
  L5_mean    <- L5_series[L5_start + 1]
  M10_mean   <- M10_series[M10_start + 1]

  # Relative Amplitude
  RA <- if (is.finite(L5_mean) && is.finite(M10_mean) && (L5_mean + M10_mean) > 0)
    (M10_mean - L5_mean) / (M10_mean + L5_mean) else NA_real_

  # IS (Interdaily Stability) and IV (Intradaily Variability) on the full series
  xbar <- mean(x, na.rm = TRUE)

  # IS: variance of hourly means across the day, normalized by total variance
  minute_of_day <- (seq_along(x) - 1) %% 1440
  hour_of_day   <- minute_of_day %/% 60L                 # 0..23
  hr_means <- tapply(x, hour_of_day, function(v) mean(v, na.rm = TRUE))
  # ensure length 24
  hr24 <- rep(NA_real_, 24L); hr24[as.integer(names(hr_means)) + 1L] <- as.numeric(hr_means)

  n_eff <- sum(is.finite(x))
  denom_IS <- 24 * sum((x - xbar)^2, na.rm = TRUE)
  IS <- if (denom_IS > 0 && n_eff > 0)
    (n_eff * sum((hr24 - xbar)^2, na.rm = TRUE)) / denom_IS else NA_real_

  # IV: fragmentation via successive 1-minute differences
  valid <- is.finite(x)
  vp <- which(valid[-length(x)] & valid[-1])
  num_IV <- sum((x[vp + 1] - x[vp])^2, na.rm = TRUE)
  den_IV <- sum((x[valid] - xbar)^2,    na.rm = TRUE)
  m_pairs <- length(vp); n_valid <- sum(valid)
  IV <- if (den_IV > 0 && m_pairs > 0)
    (n_valid * num_IV) / (m_pairs * den_IV) else NA_real_

  list(
    L5_mean        = as.numeric(L5_mean),
    L5_start_min   = as.integer(L5_start),     #
    M10_mean       = as.numeric(M10_mean),
    M10_start_min  = as.integer(M10_start),    #
    RA             = as.numeric(RA),
    IS             = as.numeric(IS),
    IV             = as.numeric(IV),
    profile_24h    = prof                      # 1440-length average day (for plotting/checks)
  )
}
