#' reducePeaks
#'
#' This function is used to reduce the peaks within a feature distribution
#' so that the models can be fitted properly. The flexmix package can struggle
#' to converge on a solution if there are large spikes in the distribution. 
#' @param toReduce Input feature distribution
#' @return Returns the input feature with the peaks reduced
reducePeaks = function(toReduce) {
  reduction = toReduce #Makes copy of input
  
  seenVals = unique(toReduce)
  
  #Checks if it is a discrete variable and doesn't use hist
  if(length(seenVals)/length(toReduce) < 0.01){
    breaks = seq(min(seenVals),max(seenVals),1)
    counts = sapply(breaks,function(x){sum(toReduce==x)})
    breaks = c(-1,breaks)
    bins = list(breaks = breaks, counts = counts)
  } else {
    bins = graphics::hist(toReduce, breaks = "FD", plot = FALSE)
  }

  #Slightly lowers the first break so the lower bound is inclusive
  bins$breaks[1] = bins$breaks[1] - (bins$breaks[2]-bins$breaks[1])
  
  # Goes through each bin and checks to see whether the surrounding bins are
  # significantly smaller than it, and if so it lowers the number of values 
  # in that bin to be closer to the surrounding bins.
  for (i in 1:(length(bins$counts))){
    diff = 1
    bigNeighbor = max(bins$counts[c(i-diff,i+diff)],na.rm = TRUE)
    while(bigNeighbor <= 0){
      diff = diff + 1
      low = max(c(0,i-diff))
      high = min(length(bins$counts),i+diff)
      bigNeighbor = max(bins$counts[c(low,high)],na.rm = TRUE)
    }
    if (bigNeighbor > 0 & bins$counts[i] > (1.5*bigNeighbor)){
      newCount = floor(1.5*bigNeighbor)
      lower = round(bins$breaks[i],2)
      upper = round(bins$breaks[i+1],2)
      if (lower == upper){
        indices = which(reduction == lower)
      }
      else {
        indices = which((reduction > lower & reduction <= upper))
      }
      reduction = reduction[-c(sample(indices, (bins$counts[i]-newCount)))]
    }
  }
  
  return(reduction)
}
