% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/RcppExports.R
\name{bdPCA_hdf5}
\alias{bdPCA_hdf5}
\title{Principal Component Analysis for HDF5-Stored Matrices}
\usage{
bdPCA_hdf5(
  filename,
  group,
  dataset,
  ncomponents = 0L,
  bcenter = FALSE,
  bscale = FALSE,
  k = 2L,
  q = 1L,
  rankthreshold = 0,
  SVDgroup = NULL,
  overwrite = FALSE,
  method = NULL,
  threads = NULL
)
}
\arguments{
\item{filename}{Character string. Path to the HDF5 file containing the input matrix.}

\item{group}{Character string. Path to the group containing the input dataset.}

\item{dataset}{Character string. Name of the input dataset to analyze.}

\item{ncomponents}{Integer. Number of principal components to compute (default = 0,
which computes all components).}

\item{bcenter}{Logical. If TRUE, centers the data by subtracting column means.
Default is FALSE.}

\item{bscale}{Logical. If TRUE, scales the centered columns by their standard
deviations (if centered) or root mean square. Default is FALSE.}

\item{k}{Integer. Number of local SVDs to concatenate at each level (default = 2).
Controls memory usage in block computation.}

\item{q}{Integer. Number of levels for SVD computation (default = 1).
Higher values can improve accuracy but increase computation time.}

\item{rankthreshold}{Numeric. Threshold for determining matrix rank (default = 0).
Must be between 0 and 0.1.}

\item{SVDgroup}{Character string. Group name where intermediate SVD results are
stored. If SVD was previously computed, results will be reused from this group.}

\item{overwrite}{Logical. If TRUE, forces recomputation of SVD even if results exist.}

\item{method}{Character string. Computation method:
\itemize{
\item "auto": Automatically selects method based on matrix size
\item "blocks": Uses block-based computation (for large matrices)
\item "full": Performs direct computation (for smaller matrices)
}}

\item{threads}{Integer. Number of threads for parallel computation.}
}
\value{
A list containing the paths to the PCA results stored in the HDF5 file:
\describe{
\item{fn}{Character string. Path to the HDF5 file containing the results}
\item{lambda}{Character string. Dataset path to eigenvalues \eqn{\lambda}}
\item{variance}{Character string. Dataset path to variance explained by each PC}
\item{cumvar}{Character string. Dataset path to cumulative variance explained}
\item{var.coord}{Character string. Dataset path to variable coordinates on the PCs}
\item{var.cos2}{Character string. Dataset path to squared cosines (quality of representation) for variables}
\item{ind.dist}{Character string. Dataset path to distances of individuals from the origin}
\item{components}{Character string. Dataset path to principal components (rotated data)}
\item{ind.coord}{Character string. Dataset path to individual coordinates on the PCs}
\item{ind.cos2}{Character string. Dataset path to squared cosines (quality of representation) for individuals}
\item{ind.contrib}{Character string. Dataset path to contributions of individuals to each PC}
}
All results are written to the HDF5 file in the group 'PCA/\code{dataset}'.
}
\description{
Performs Principal Component Analysis (PCA) on a large matrix stored in an HDF5 file.
PCA reduces the dimensionality of the data while preserving as much variance as
possible. The implementation uses SVD internally for efficient and numerically
stable computation.
}
\details{
This function implements a scalable PCA algorithm suitable for large matrices
that may not fit in memory. Key features include:
\itemize{
\item Automatic method selection based on matrix size
\item Block-based computation for large matrices
\item Optional data preprocessing (centering and scaling)
\item Parallel processing support
\item Memory-efficient incremental algorithm
\item Reuse of existing SVD results
}

The implementation uses SVD internally and supports two computation methods:
\itemize{
\item Full decomposition: Suitable for matrices that fit in memory
\item Block-based decomposition: For large matrices, uses an incremental algorithm
}
}
\examples{
\dontrun{
# Create a sample large matrix in HDF5
library(rhdf5)
X <- matrix(rnorm(10000), 1000, 10)
h5createFile("data.h5")
h5write(X, "data.h5", "data/matrix")

# Basic PCA with default parameters
bdPCA_hdf5("data.h5", "data", "matrix")

# PCA with preprocessing and specific number of components
bdPCA_hdf5("data.h5", "data", "matrix",
           ncomponents = 3,
           bcenter = TRUE, bscale = TRUE,
           method = "blocks",
           threads = 4)
}

}
\references{
\itemize{
\item Halko, N., Martinsson, P. G., & Tropp, J. A. (2011). Finding structure with randomness:
Probabilistic algorithms for constructing approximate matrix decompositions.
SIAM Review, 53(2), 217-288.
\item Jolliffe, I. T. (2002). Principal Component Analysis, Second Edition.
Springer Series in Statistics.
}
}
\seealso{
\itemize{
\item \code{\link{bdSVD_hdf5}} for the underlying SVD computation
\item \code{\link{bdNormalize_hdf5}} for data preprocessing options
}
}
