% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/1_the_BCS_distributions.R
\name{BCS}
\alias{BCS}
\alias{dBCS}
\alias{pBCS}
\alias{qBCS}
\alias{rBCS}
\title{The Box-Cox Symmetric Distributions}
\usage{
dBCS(x, mu, sigma, lambda, zeta, family = "NO", log = FALSE)

pBCS(
  q,
  mu,
  sigma,
  lambda,
  zeta,
  family = "NO",
  lower.tail = TRUE,
  log.p = FALSE
)

qBCS(
  p,
  mu,
  sigma,
  lambda,
  zeta,
  family = "NO",
  lower.tail = TRUE,
  log.p = FALSE
)

rBCS(n, mu, sigma, lambda, zeta, family = "NO")
}
\arguments{
\item{x, q}{vector of positive quantiles.}

\item{mu}{vector of strictly positive scale parameters.}

\item{sigma}{vector of strictly positive relative dispersion parameters.}

\item{lambda}{vector of real-valued skewness parameters. If \code{lambda = 0}, the BCS
distribution reduces to the corresponding log-symmetric distribution with parameters
\code{mu} and \code{sigma} (and a possible extra parameter \code{zeta}).}

\item{zeta}{strictly positive extra parameter. It must be specified with only one value
in cases where the BCS distribution has an extra parameter. See “Details” below.}

\item{family}{a character that specifies the symmetric generating family of the BCS distribution.
Available options are: \code{"NO"} (default), \code{"ST"}, \code{"LOI"}, \code{"LOII"},
\code{"PE"}, \code{"SN"}, \code{"HP"}, and \code{"SL"}, corresponding to the normal,
Student-\emph{t}, type I logistic, type II logistic, power exponential, sinh-normal,
hyperbolic, and slash distributions, respectively.}

\item{log, log.p}{logical; if \code{TRUE}, probabilities \code{p} are given as \code{log(p)}.
Default is \code{FALSE}.}

\item{lower.tail}{logical; if \code{TRUE} (default), probabilities are
\eqn{P(X \le x)} otherwise, \eqn{P(X > x)}.}

\item{p}{vector of probabilities.}

\item{n}{number of observations. If `n` is a vector, its length is used as the number of
required observations.}
}
\value{
\code{dBCS} returns the density function, \code{pBCS} gives the cumulative distribution function,
\code{qBCS} provides the quantile function, and \code{rBCS} generates random variables.
}
\description{
Density function, distribution function, quantile function, and random generation
    for the class of the Box-Cox symmetric (BCS) distributions.
}
\details{
The class of the BCS distributions was introduced by Ferrari and
    Fumes (2017). It consists of a broad class of probability models for
    positive continuous data, which includes flexible distributions with
    different levels of skewness and tail-heaviness.

    The BCS class includes, as special cases, the Box-Cox \emph{t} (Rigby and Stasinopoulos, 2006),
    Box-Cox normal (or Box-Cox Cole-Green; Cole and Green, 1992), Box-Cox power exponential
    (Rigby and Stasinopoulos, 2004) distributions, as well as the log-symmetric
    distributions (Vanegas and Paula, 2016).

    Let \eqn{Y} be a positive continuous random variable with a BCS distribution
    with parameters \eqn{\mu > 0}, \eqn{\sigma > 0}, and \eqn{\lambda \in \mathbb{R}}
    and density generating function \eqn{r}. The probability density function of
    \eqn{Y} is given by

    \eqn{
    f(y; \mu, \sigma, \lambda) = \left\{\begin{array}{ll}
      \dfrac{y^{\lambda-1}}{\mu^\lambda \sigma} \dfrac{r(z^2)}{R\left(\frac{1}{\sigma |\lambda|}\right)}, & \mbox{ if } \lambda \neq 0,\\
      \dfrac{1}{y\sigma} r(z^2), & \mbox{ if } \lambda = 0,
      \end{array}\right., \quad y > 0,
    }

    where
    \eqn{
    z = \left\{
    \begin{array}{ll}
    \dfrac{1}{\sigma \lambda} \left\{\left(\frac{y}{\mu}\right)^\lambda - 1 \right\}, & \mbox{ if } \lambda \neq 0, \\
    \dfrac{1}{\sigma} \log\left(\frac{y}{\mu}\right), & \mbox{ if } \lambda = 0,
    \end{array}
    \right.
    }

    \eqn{r:[0,\infty) \longrightarrow [0, \infty)}
    satisfies \eqn{\int_0^\infty u^{-1/2}r(u)\textrm{d} u = 1}, and
    \eqn{R(x) = \int_{-\infty}^x r(u^2)\textrm{d} u, x \in \mathbb{R}}.

    The function \eqn{r} is called density generating function, and it specifies the
    generating symmetric family of \eqn{Y} within the class of the BCS probability
    models. This function can also depend on extra parameters, such as the Box-Cox \emph{t} and
    Box-Cox power exponential distributions. We call these extra parameters
    \code{zeta}. The currently available BCS distributions in the \code{BCSreg} package
    are listed below:
    \tabular{llc}{
    \bold{Distribution}  \tab \bold{Family abbreviation} \tab \bold{N. of extra parameters}\cr
    Box-Cox Hyperbolic  \tab \code{"HP"}      \tab  1  \cr
    Box-Cox Type I Logistic  \tab \code{"LOI"}      \tab  0  \cr
    Box-Cox Type II Logistic  \tab \code{"LOII"}      \tab  0  \cr
    Box-Cox Normal  \tab \code{"NO"}      \tab  0  \cr
    Box-Cox Power Exponential  \tab \code{"PE"}      \tab  1  \cr
    Box-Cox Sinh-Normal  \tab \code{"SN"}      \tab  1  \cr
    Box-Cox Slash  \tab \code{"SL"}      \tab  1  \cr
    Box-Cox \emph{t}  \tab \code{"ST"}      \tab  1  \cr
    }
}
\examples{
# Probability density function

## Right-skewed distributions
curve(dBCS(x, 3, 0.3, -1.5, family = "NO"), xlim = c(0, 7), ylim = c(0, 0.7), ylab = "Density")
curve(dBCS(x, 3, 0.3, -1.5, 4, family = "ST"), add = TRUE, col = 2)
curve(dBCS(x, 3, 0.3, -1.5, 5, family = "PE"), add = TRUE, col = 4)
legend("topright", legend = c("BCNO", "BCT", "BCPE"), lty = 1, col = c(1, 2, 4))

## Truncated symmetric distributions (with support on (0, Inf))
curve(dBCS(x, 3, 0.3, 1, family = "NO"), xlim = c(0, 7), ylim = c(0, 0.7), ylab = "Density")
curve(dBCS(x, 3, 0.3, 1, 4, family = "ST"), add = TRUE, col = 2)
curve(dBCS(x, 3, 0.3, 1, 5, family = "PE"), add = TRUE, col = 4)
legend("topright", legend = c("BCNO", "BCT", "BCPE"), lty = 1, col = c(1, 2, 4))

## Left-skewed distributions
curve(dBCS(x, 3, 0.3, 3, family = "NO"), xlim = c(0, 7), ylim = c(0, 0.7), ylab = "Density")
curve(dBCS(x, 3, 0.3, 3, 4, family = "ST"), add = TRUE, col = 2)
curve(dBCS(x, 3, 0.3, 3, 5, family = "PE"), add = TRUE, col = 4)
legend("topright", legend = c("BCNO", "BCT", "BCPE"), lty = 1, col = c(1, 2, 4))

# Random generation

## Parameter setting
mu <- 5        # scale parameter
sigma <- 0.2   # relative dispersion parameter
lambda <- -0.2 # skewness parameter

## Generating family
family <- "NO"

## Visualization
x <- rBCS(10000, mu, sigma, lambda, family = family)

hist(x, prob = TRUE, col = "white", main = "")
curve(dBCS(x, mu, sigma, lambda, family = family), col = "blue", add = TRUE)

plot(ecdf(x), main = "")
curve(pBCS(x, mu, sigma, lambda, family = family), col = "blue", add = TRUE)
}
\references{
Cole, T., and Green, P.J. (1992). Smoothing reference centile curves: the LMS
     method and penalized likelihood. \emph{Statistics in medicine}, 11, 1305-1319.

 Ferrari, S. L., and Fumes, G. (2017). Box-Cox symmetric distributions and
     applications to nutritional data. \emph{AStA Advances in Statistical Analysis}, 101, 321-344.

 Medeiros, R. M. R., and Queiroz, F. F. (2025). Flexible modeling of nonnegative continuous
     data: Box-Cox symmetric regression and its zero-adjusted extension.

 Rigby, R. A., and Stasinopoulos, D. M. (2004). Smooth centile curves for skew
     and kurtotic data modelled using the Box-Cox power exponential
     distribution. \emph{Statistics in medicine}, 23, 3053-3076.

 Rigby, R. A., and Stasinopoulos, D. M. (2006). Using the Box-Cox t
     distribution in GAMLSS to model skewness and kurtosis. \emph{Statistical Modelling}, 6, 209-229.

 Vanegas, L. H., and Paula, G. A. (2016). Log-symmetric distributions:
     statistical properties and parameter estimation. \emph{Brazilian Journal of Probability and Statistics}, 30, 196-220.
}
\seealso{
\code{\link{ZABCS}} to access the density function, distribution
    function, quantile function, and a random number generator for the
    zero-adjusted BCS distributions. \code{\link{BCSreg}} for estimating the
    parameters of a BCS regression model.
}
\author{
Francisco F. de Queiroz <\email{felipeq@ime.usp.br}>

Rodrigo M. R. de Medeiros <\email{rodrigo.matheus@ufrn.br}>
}
