% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/finite-diff.R
\name{fdCoef}
\alias{fdCoef}
\title{Finite-difference coefficients for arbitrary grids}
\usage{
fdCoef(
  deriv.order = 1L,
  side = c(0L, 1L, -1L),
  acc.order = 2L,
  stencil = NULL,
  zero.action = c("drop", "round", "none"),
  zero.tol = NULL
)
}
\arguments{
\item{deriv.order}{Order of the derivative (\eqn{m}{m} in \eqn{\frac{d^m f}{dx^m}}{d^m/dx^m f(x)})
for which a numerical approximation is needed.}

\item{side}{Integer that determines the type of finite-difference scheme:
\code{0} for central (AKA symmetrical or two-sided; the default),
\code{1} for forward, and \code{-1} for backward.
Using \code{2} (for 'two-sided') triggers a warning and is treated as \code{0}.
with a warning. Unless the function is computationally prohibitively,
central differences are strongly recommended for their accuracy.}

\item{acc.order}{Order of accuracy: defines how the approximation error scales
with the step size \eqn{h}{h}, specifically \eqn{O(h^{a+1})}{O(h^(a+1))}, where
\eqn{a}{a} is the accuracy order and depends on the higher-order derivatives
of the function.}

\item{stencil}{Optional custom vector of points for function evaluation.
Must include at least \code{m+1} points for the \code{m}-th order derivative.}

\item{zero.action}{Character string specifying how to handle near-zero weights:
\code{"drop"} to omit small (less in absolute value than \code{zero.tol} times
the median weight) weights and corresponding stencil points, \code{"round"}
to round small weights to zero, and \code{"none"} to leave
all weights as calculated.
E.g. the stencil for \eqn{f'(x)}{f'(x)} is \code{(-1, 0, 1)} with weights
\code{(-0.5, 0, 0.5)}; using \code{"drop"} eliminates the zero weight,
and the redundant \code{f(x)} is not computed.}

\item{zero.tol}{Non-negative scalar defining the threshold: weights below
\code{zero.tol} times the median weight are considered near-zero.}
}
\value{
A list containing the \code{stencil} used and the corresponding
\code{weights} for each point.
}
\description{
This function computes the coefficients for a numerical approximation to derivatives
of any specified order. It provides the minimally sufficient stencil
for the chosen derivative order and desired accuracy order.
It can also use any user-supplied stencil (uniform or non-uniform).
For that stencil \eqn{\{b_i\}_{i=1}^n}{{b[i]}, i = 1, ..., n}, it computes
the optimal weights \eqn{\{w_i\}}{{w[i]}} that yield
the numerical approximation of the derivative:
\deqn{\frac{d^m f}{dx^m} \approx h^{-m} \sum_{i=1}^n w_i f(x + b_i\cdot h)}{d^m/dx^m f(x) ~ sum_i w[i] f(x + b[i]*h)}
}
\details{
This function relies on the approach of approximating numerical derivarives
by weghted sums of function values described in \insertCite{fornberg1988generation}{pnd}.
It reproduces all tables from this paper exactly; see the example below to
create Table 1.

The finite-difference coefficients for any given stencil are given as a solution of a linear
system. The capabilities of this function are similar to those of \insertCite{taylor2016finite}{pnd},
but instead of matrix inversion, the \insertCite{bjorck1970solution}{pnd} algorithm is used because
the left-hand-side matrix is a Vandermonde matrix, and its inverse may be
very inaccurate, especially for long one-sided stencils.

The weights computed for the stencil via this algorithm are very reliable; numerical
simulations in \insertCite{higham1987error}{pnd} show that the relative error is
low even for ill-conditioned systems. \insertCite{kostyrka2025what}{pnd}
computes the exact relative error of the weights on the stencils returned by
this function; the zero tolerance is based on these calculations.
}
\examples{
fdCoef()  # Simple two-sided derivative
fdCoef(2) # Simple two-sided second derivative
fdCoef(acc.order = 4)$weights * 12  # Should be (1, -8, 8, -1)

# Using an custom stencil for the first derivative: x-2h and x+h
fdCoef(stencil = c(-2, 1), acc.order = 1)

# Reproducing Table 1 from Fornberg (1988) (cited above)
pad9 <- function(x) {l <- length(x); c(a <- rep(0, (9-l)/2), x, a)}
f <- function(d, a) pad9(fdCoef(deriv.order = d, acc.order = a,
                                zero.action = "round")$weights)
t11 <- t(sapply((1:4)*2, function(a) f(d = 1, a)))
t12 <- t(sapply((1:4)*2, function(a) f(d = 2, a)))
t13 <- t(sapply((1:3)*2, function(a) f(d = 3, a)))
t14 <- t(sapply((1:3)*2, function(a) f(d = 4, a)))
t11 <- cbind(t11[, 1:4], 0, t11[, 5:8])
t13 <- cbind(t13[, 1:4], 0, t13[, 5:8])
t1 <- data.frame(OrdDer = rep(1:4, times = c(4, 4, 3, 3)),
                 OrdAcc = c((1:4)*2, (1:4)*2, (1:3)*2, (1:3)*2),
                 rbind(t11, t12, t13, t14))
colnames(t1)[3:11] <- as.character(-4:4)
print(t1, digits = 4)
}
\references{
\insertAllCited{}
}
