% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tax_range_space.R
\name{tax_range_space}
\alias{tax_range_space}
\title{Calculate the geographic range of fossil taxa}
\usage{
tax_range_space(
  occdf,
  name = "genus",
  lng = "lng",
  lat = "lat",
  method = "lat",
  spacing = 100
)
}
\arguments{
\item{occdf}{\code{dataframe}. A dataframe of fossil occurrences.
This dataframe should contain at least three columns: names of taxa,
longitude and latitude (see \code{name}, \code{lng}, and \code{lat} arguments).}

\item{name}{\code{character}. The name of the column you wish to be treated
as the input names (e.g. "species" or "genus"). NA data should be removed
prior to function call.}

\item{lng}{\code{character}. The name of the column you wish to be treated
as the input longitude (e.g. "lng" or "p_lng"). NA data should be removed
prior to function call.}

\item{lat}{\code{character}. The name of the column you wish to be treated
as the input latitude (e.g. "lat" or "p_lat"). NA data should be removed
prior to function call.}

\item{method}{\code{character}. How should geographic range be calculated
for each taxon in \code{occdf}? Four options exist in this function:
"con", "lat", "gcd", and "occ". See Details for a description of each.}

\item{spacing}{\code{numeric}. The desired spacing (in km) between the
center of adjacent grid cells. Only required if the \code{method} argument is set
to "occ". The default is 100.}
}
\value{
A \code{dataframe} with method-specific columns:
\itemize{
\item For the "con" method, a \code{dataframe} with each unique taxa (\code{taxon})
and taxon ID (\code{taxon_id}) by convex hull coordinate (\code{lng} & \code{lat})
combination, and area (\code{area}) in
km\ifelse{html}{\out{<sup>2</sup>}}{\eqn{^2}} is returned.
\item For the "lat" method, a \code{dataframe} with unique taxa (\code{taxon}),
taxon ID (\code{taxon_id}), maximum latitude of occurrence (\code{max_lat}),
minimum latitude of occurrence (\code{min_lat}), and latitudinal
range (\code{range_lat}) is returned.
\item For the "gcd" method, a \code{dataframe} with each unique taxa (\code{taxon})
and taxon ID (\code{taxon_id}) by coordinate combination (\code{lng} & \code{lat}) of the
two most distant points, and the Great Circle Distance (\code{gcd}) between
these points in km is returned.
\item For the "occ" method, a \code{dataframe} with unique taxa (\code{taxon}), taxon
ID (\code{taxon_id}), the number of occupied cells (\code{n_cells}), proportion of
occupied cells from all occupied by occurrences (\code{proportional_occ}),
and the spacing between cells (\code{spacing}) in km is returned. Note: the number
of occupied cells and proportion of occupied cells is highly dependent on
the user-defined \code{spacing.}
For the "con", "lat" and "gcd" method, values of zero indicate that the
respective taxon is a singleton (i.e. represented by only one occurrence).
}
}
\description{
A function to calculate the geographic range of fossil taxa from occurrence
data. The function can calculate geographic range in four ways: convex hull,
latitudinal range, maximum Great Circle Distance, and the number of
occupied equal-area hexagonal grid cells.
}
\details{
Four commonly applied approaches (Darroch et al. 2020)
are available using the \code{tax_range_space} function for calculating ranges:
\itemize{
\item Convex hull: the "con" method calculates the geographic range of taxa
using a convex hull for each taxon in \code{occdf}, and calculates the area of
the convex hull (in km\ifelse{html}{\out{<sup>2</sup>}}{\eqn{^2}}) using
\code{\link[geosphere:areaPolygon]{geosphere::areaPolygon()}}. The
convex hull method works by creating a polygon that encompasses all
occurrence points of the taxon.
\item Latitudinal: the "lat" method calculates the palaeolatitudinal
range of a taxon. It does so for each taxon in \code{occdf} by finding their
maximum and minimum latitudinal occurrence (from input \code{lat}).
The palaeolatitudinal range of each taxon is also calculated (i.e. the
difference between the minimum and maximum latitude).
\item Maximum Great Circle Distance: the "gcd" method calculates the maximum
Great Circle Distance between occurrences for each taxon in \code{occdf}. It does
so using \code{\link[geosphere:distHaversine]{geosphere::distHaversine()}}.
This function calculates Great Circle Distance using the Haversine method
with the radius of the Earth set to the 6378.137 km.
Great Circle Distance represents the shortest distance between two
points on the surface of a sphere. This is different from Euclidean Distance,
which represents the distance between two points on a plane.
\item Occupied cells: the "occ" method calculates the number and proportion of
occupied equal-area grid cells. It does so using discrete hexagonal grids
via the \code{\link[h3jsr]{h3jsr}} package. This package relies on
\href{https://h3geo.org/docs/}{Uber's H3} library, a geospatial indexing system
that partitions the world into hexagonal cells. In H3, 16 different
resolutions are available
(\href{https://h3geo.org/docs/core-library/restable/}{see here}).
In the implementation of the \code{tax_range_space()} function, the resolution is
defined by the user-input \code{spacing} which represents the distance between
the centroid of adjacent cells. Using this distance, the function identifies
which resolution is most similar to the input \code{spacing}, and uses this
resolution.
}
}
\section{Reference(s)}{

Darroch, S. A., Casey, M. M., Antell, G. S., Sweeney, A., & Saupe, E. E.
(2020). High preservation potential of paleogeographic range size
distributions in deep time. The American Naturalist, 196(4), 454-471.
}

\section{Developer(s)}{

Lewis A. Jones
}

\section{Reviewer(s)}{

Bethany Allen & Christopher D. Dean
}

\examples{
# Grab internal data
occdf <- tetrapods[1:100, ]
# Remove NAs
occdf <- subset(occdf, !is.na(genus))
# Convex hull
ex1 <- tax_range_space(occdf = occdf, name = "genus", method = "con")
# Latitudinal range
ex2 <- tax_range_space(occdf = occdf, name = "genus", method = "lat")
# Great Circle Distance
ex3 <- tax_range_space(occdf = occdf, name = "genus", method = "gcd")
# Occupied grid cells
ex4 <- tax_range_space(occdf = occdf, name = "genus",
                       method = "occ", spacing = 500)
}
