/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5private.h"          
#include "H5Eprivate.h"         
#include "H5Fprivate.h"         
#include "H5HGprivate.h"        
#include "H5VLnative_private.h" 

herr_t
H5VL__native_blob_put(void *obj, const void *buf, size_t size, void *blob_id, void H5_ATTR_UNUSED *ctx)
{
    H5F_t   *f  = (H5F_t *)obj;       
    uint8_t *id = (uint8_t *)blob_id; 
    H5HG_t   hobjid;                  
    herr_t   ret_value = SUCCEED;     

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(size == 0 || buf);
    assert(id);

    
    if (H5HG_insert(f, size, buf, &hobjid) < 0)
        HGOTO_ERROR(H5E_VOL, H5E_WRITEERROR, FAIL, "unable to write blob information");

    
    H5F_addr_encode(f, &id, hobjid.addr);
    UINT32ENCODE(id, hobjid.idx);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_blob_get(void *obj, const void *blob_id, void *buf, size_t size, void H5_ATTR_UNUSED *ctx)
{
    H5F_t         *f  = (H5F_t *)obj;             
    const uint8_t *id = (const uint8_t *)blob_id; 
    H5HG_t         hobjid;                        
    size_t         hobj_size = 0;                 
    herr_t         ret_value = SUCCEED;           

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(id);
    assert(buf);

    
    H5F_addr_decode(f, &id, &hobjid.addr);
    UINT32DECODE(id, hobjid.idx);

    
    if (hobjid.addr > 0) {
        
        if (H5HG_get_obj_size(f, &hobjid, &hobj_size) < 0)
            HGOTO_ERROR(H5E_VOL, H5E_CANTGETSIZE, FAIL, "can't get object size");
        if (hobj_size != size)
            HGOTO_ERROR(H5E_VOL, H5E_BADSIZE, FAIL, "Expected global heap object size does not match");

        
        if (NULL == H5HG_read(f, &hobjid, buf, &hobj_size))
            HGOTO_ERROR(H5E_VOL, H5E_READERROR, FAIL, "unable to read VL information");
    }

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5VL__native_blob_specific(void *obj, void *blob_id, H5VL_blob_specific_args_t *args)
{
    H5F_t *f         = (H5F_t *)obj; 
    herr_t ret_value = SUCCEED;      

    FUNC_ENTER_PACKAGE

    
    assert(f);
    assert(blob_id);

    switch (args->op_type) {
        case H5VL_BLOB_ISNULL: {
            const uint8_t *id = (const uint8_t *)blob_id; 
            haddr_t        addr;                          

            
            H5F_addr_decode(f, &id, &addr);

            
            *args->args.is_null.isnull = (addr == 0 ? true : false);

            break;
        }

        case H5VL_BLOB_SETNULL: {
            uint8_t *id = (uint8_t *)blob_id; 

            
            H5F_addr_encode(f, &id, (haddr_t)0);
            UINT32ENCODE(id, 0);

            break;
        }

        case H5VL_BLOB_DELETE: {
            const uint8_t *id = (const uint8_t *)blob_id; 
            H5HG_t         hobjid;                        

            
            H5F_addr_decode(f, &id, &hobjid.addr);
            UINT32DECODE(id, hobjid.idx);

            
            if (hobjid.addr > 0)
                if (H5HG_remove(f, &hobjid) < 0)
                    HGOTO_ERROR(H5E_VOL, H5E_CANTREMOVE, FAIL, "unable to remove heap object");

            break;
        }

        default:
            HGOTO_ERROR(H5E_VOL, H5E_UNSUPPORTED, FAIL, "invalid specific operation");
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
