/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#include "H5CXmodule.h" 

#include "H5private.h"   
#include "H5CXprivate.h" 
#include "H5Dprivate.h"  
#include "H5Eprivate.h"  
#include "H5FLprivate.h" 
#include "H5Iprivate.h"  
#include "H5Lprivate.h"  
#include "H5MMprivate.h" 
#include "H5Pprivate.h"  

#ifdef H5_HAVE_THREADSAFE_API

#define H5CX_get_my_context() H5TS_get_api_ctx_ptr()
#else 

#define H5CX_get_my_context() (&H5CX_head_g)
#endif 

#define H5CX_RETRIEVE_PLIST(PL, FAILVAL)                                                                     \
                                                        \
    if (NULL == (*head)->ctx.PL)                                                                             \
                                                                          \
        if (H5_UNLIKELY(NULL ==                                                                              \
                        ((*head)->ctx.PL = (H5P_genplist_t *)H5I_object((*head)->ctx.H5_GLUE(PL, _id)))))    \
            HGOTO_ERROR(H5E_CONTEXT, H5E_BADTYPE, (FAILVAL), "can't get property list");

#define H5CX_RETRIEVE_PROP_COMMON(PL, DEF_PL, PROP_NAME, PROP_FIELD)                                         \
    {                                                                                                        \
                                                                                                             \
                                                                        \
        if ((*head)->ctx.H5_GLUE(PL, _id) == (DEF_PL))                                                       \
            H5MM_memcpy(&(*head)->ctx.PROP_FIELD, &H5_GLUE3(H5CX_def_, PL, _cache).PROP_FIELD,               \
                        sizeof(H5_GLUE3(H5CX_def_, PL, _cache).PROP_FIELD));                                 \
        else {                                                                                               \
                                                                             \
            H5CX_RETRIEVE_PLIST(PL, FAIL)                                                                    \
                                                                                                             \
                                                                                       \
            if (H5_UNLIKELY(H5P_get((*head)->ctx.PL, (PROP_NAME), &(*head)->ctx.PROP_FIELD) < 0))            \
                HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "can't retrieve value from API context");        \
        }                                                                                      \
                                                                                                             \
                                                                                \
        (*head)->ctx.H5_GLUE(PROP_FIELD, _valid) = true;                                                     \
    }

#define H5CX_RETRIEVE_PROP_VALID(PL, DEF_PL, PROP_NAME, PROP_FIELD)                                          \
                                                          \
    if (!(*head)->ctx.H5_GLUE(PROP_FIELD, _valid))                                                           \
    H5CX_RETRIEVE_PROP_COMMON(PL, DEF_PL, PROP_NAME, PROP_FIELD)

#define H5CX_RETRIEVE_PROP_VALID_SET(PL, DEF_PL, PROP_NAME, PROP_FIELD)                                      \
                                                          \
    if (!((*head)->ctx.H5_GLUE(PROP_FIELD, _valid) || (*head)->ctx.H5_GLUE(PROP_FIELD, _set)))               \
    H5CX_RETRIEVE_PROP_COMMON(PL, DEF_PL, PROP_NAME, PROP_FIELD)

#if defined(H5_HAVE_PARALLEL) && defined(H5_HAVE_INSTRUMENTED_LIBRARY)

#define H5CX_TEST_SET_PROP(PROP_NAME, PROP_FIELD)                                                            \
    {                                                                                                        \
        htri_t check_prop = 0;                    \
                                                                                                             \
                                                                       \
        if (!(*head)->ctx.H5_GLUE(PROP_FIELD, _set)) {                                                       \
                                                            \
            H5CX_RETRIEVE_PLIST(dxpl, FAIL)                                                                  \
                                                                                                             \
            if (H5_UNLIKELY((check_prop = H5P_exist_plist((*head)->ctx.dxpl, PROP_NAME)) < 0))               \
                HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "error checking for property");                  \
        }                                                                                        \
                                                                                                             \
                                       \
        if ((*head)->ctx.H5_GLUE(PROP_FIELD, _set) || check_prop > 0) {                                      \
                               \
            (*head)->ctx.PROP_FIELD                = PROP_FIELD;                                             \
            (*head)->ctx.H5_GLUE(PROP_FIELD, _set) = true;                                                   \
        }                                                                                        \
    }
#endif 

#define H5CX_SET_PROP(PROP_NAME, PROP_FIELD)                                                                 \
    if ((*head)->ctx.H5_GLUE(PROP_FIELD, _set)) {                                                            \
                                                            \
        H5CX_RETRIEVE_PLIST(dxpl, FAIL)                                                                      \
                                                                                                             \
                                                                                       \
        if (H5_UNLIKELY(H5P_set((*head)->ctx.dxpl, PROP_NAME, &(*head)->ctx.PROP_FIELD) < 0))                \
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTSET, FAIL, "error setting data xfer property");                 \
    } 

typedef struct H5CX_dxpl_cache_t {
    size_t    max_temp_buf;         
    void     *tconv_buf;            
    void     *bkgr_buf;             
    H5T_bkg_t bkgr_buf_type;        
    double    btree_split_ratio[3]; 
    size_t    vec_size;             
#ifdef H5_HAVE_PARALLEL
    H5FD_mpio_xfer_t io_xfer_mode; 
    H5FD_mpio_collective_opt_t mpio_coll_opt; 
    uint32_t mpio_local_no_coll_cause;        
    uint32_t mpio_global_no_coll_cause;       
    H5FD_mpio_chunk_opt_t
             mpio_chunk_opt_mode;       
    unsigned mpio_chunk_opt_num;        
    unsigned mpio_chunk_opt_ratio;      
#endif                                  
    H5Z_EDC_t               err_detect; 
    H5Z_cb_t                filter_cb;  
    H5Z_data_xform_t       *data_transform;        
    H5T_vlen_alloc_info_t   vl_alloc_info;         
    H5T_conv_cb_t           dt_conv_cb;            
    H5D_selection_io_mode_t selection_io_mode;     
    uint32_t                no_selection_io_cause; 
    uint32_t actual_selection_io_mode;             
    bool modify_write_buf;                         
} H5CX_dxpl_cache_t;

typedef struct H5CX_lcpl_cache_t {
    H5T_cset_t encoding;           
    unsigned   intermediate_group; 
} H5CX_lcpl_cache_t;

typedef struct H5CX_lapl_cache_t {
    size_t nlinks; 
} H5CX_lapl_cache_t;

typedef struct H5CX_dcpl_cache_t {
    bool    do_min_dset_ohdr; 
    uint8_t ohdr_flags;       
} H5CX_dcpl_cache_t;

typedef struct H5CX_dapl_cache_t {
    const char *extfile_prefix; 
    const char *vds_prefix;     
} H5CX_dapl_cache_t;

typedef struct H5CX_fapl_cache_t {
    H5F_libver_t low_bound;  
    H5F_libver_t high_bound; 
} H5CX_fapl_cache_t;

bool H5_PKG_INIT_VAR = false;

#ifndef H5_HAVE_THREADSAFE_API
static H5CX_node_t *H5CX_head_g = NULL; 
#endif                                  

static H5CX_dxpl_cache_t H5CX_def_dxpl_cache;

static H5CX_lcpl_cache_t H5CX_def_lcpl_cache;

static H5CX_lapl_cache_t H5CX_def_lapl_cache;

static H5CX_dcpl_cache_t H5CX_def_dcpl_cache;

static H5CX_dapl_cache_t H5CX_def_dapl_cache;

static H5CX_fapl_cache_t H5CX_def_fapl_cache;

H5FL_DEFINE_STATIC(H5CX_state_t);

herr_t
H5CX__init_package(void)
{
    H5P_genplist_t *dx_plist;            
    H5P_genplist_t *lc_plist;            
    H5P_genplist_t *la_plist;            
    H5P_genplist_t *dc_plist;            
    H5P_genplist_t *da_plist;            
    H5P_genplist_t *fa_plist;            
    herr_t          ret_value = SUCCEED; 

    FUNC_ENTER_PACKAGE

    
    memset(&H5CX_def_dxpl_cache, 0, sizeof(H5CX_dxpl_cache_t));

    

    
    if (NULL == (dx_plist = (H5P_genplist_t *)H5I_object(H5P_DATASET_XFER_DEFAULT)))
        HGOTO_ERROR(H5E_CONTEXT, H5E_BADTYPE, FAIL, "not a dataset transfer property list");

    
    if (H5P_get(dx_plist, H5D_XFER_BTREE_SPLIT_RATIO_NAME, &H5CX_def_dxpl_cache.btree_split_ratio) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve B-tree split ratios");

    
    if (H5P_get(dx_plist, H5D_XFER_MAX_TEMP_BUF_NAME, &H5CX_def_dxpl_cache.max_temp_buf) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve maximum temporary buffer size");

    
    if (H5P_get(dx_plist, H5D_XFER_TCONV_BUF_NAME, &H5CX_def_dxpl_cache.tconv_buf) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve temporary buffer pointer");

    
    if (H5P_get(dx_plist, H5D_XFER_BKGR_BUF_NAME, &H5CX_def_dxpl_cache.bkgr_buf) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve background buffer pointer");

    
    if (H5P_get(dx_plist, H5D_XFER_BKGR_BUF_TYPE_NAME, &H5CX_def_dxpl_cache.bkgr_buf_type) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve background buffer type");

    
    if (H5P_get(dx_plist, H5D_XFER_HYPER_VECTOR_SIZE_NAME, &H5CX_def_dxpl_cache.vec_size) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve I/O vector size");

#ifdef H5_HAVE_PARALLEL
    
    if (H5P_get(dx_plist, H5D_XFER_IO_XFER_MODE_NAME, &H5CX_def_dxpl_cache.io_xfer_mode) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve parallel transfer method");
    if (H5P_get(dx_plist, H5D_XFER_MPIO_COLLECTIVE_OPT_NAME, &H5CX_def_dxpl_cache.mpio_coll_opt) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve collective transfer option");
    if (H5P_get(dx_plist, H5D_XFER_MPIO_CHUNK_OPT_HARD_NAME, &H5CX_def_dxpl_cache.mpio_chunk_opt_mode) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve chunk optimization option");
    if (H5P_get(dx_plist, H5D_XFER_MPIO_CHUNK_OPT_NUM_NAME, &H5CX_def_dxpl_cache.mpio_chunk_opt_num) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve chunk optimization threshold");
    if (H5P_get(dx_plist, H5D_XFER_MPIO_CHUNK_OPT_RATIO_NAME, &H5CX_def_dxpl_cache.mpio_chunk_opt_ratio) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve chunk optimization ratio");

    
    if (H5P_get(dx_plist, H5D_MPIO_LOCAL_NO_COLLECTIVE_CAUSE_NAME,
                &H5CX_def_dxpl_cache.mpio_local_no_coll_cause) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve local cause for breaking collective I/O");
    if (H5P_get(dx_plist, H5D_MPIO_GLOBAL_NO_COLLECTIVE_CAUSE_NAME,
                &H5CX_def_dxpl_cache.mpio_global_no_coll_cause) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL,
                    "Can't retrieve global cause for breaking collective I/O");
#endif 

    
    if (H5P_get(dx_plist, H5D_XFER_EDC_NAME, &H5CX_def_dxpl_cache.err_detect) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve error detection info");

    
    if (H5P_get(dx_plist, H5D_XFER_FILTER_CB_NAME, &H5CX_def_dxpl_cache.filter_cb) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve filter callback function");

    
    
    if (H5P_peek(dx_plist, H5D_XFER_XFORM_NAME, &H5CX_def_dxpl_cache.data_transform) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve data transform info");

    
    if (H5P_get(dx_plist, H5D_XFER_VLEN_ALLOC_NAME, &H5CX_def_dxpl_cache.vl_alloc_info.alloc_func) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve VL datatype alloc info");
    if (H5P_get(dx_plist, H5D_XFER_VLEN_ALLOC_INFO_NAME, &H5CX_def_dxpl_cache.vl_alloc_info.alloc_info) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve VL datatype alloc info");
    if (H5P_get(dx_plist, H5D_XFER_VLEN_FREE_NAME, &H5CX_def_dxpl_cache.vl_alloc_info.free_func) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve VL datatype alloc info");
    if (H5P_get(dx_plist, H5D_XFER_VLEN_FREE_INFO_NAME, &H5CX_def_dxpl_cache.vl_alloc_info.free_info) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve VL datatype alloc info");

    
    if (H5P_get(dx_plist, H5D_XFER_CONV_CB_NAME, &H5CX_def_dxpl_cache.dt_conv_cb) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve datatype conversion exception callback");

    
    if (H5P_get(dx_plist, H5D_XFER_SELECTION_IO_MODE_NAME, &H5CX_def_dxpl_cache.selection_io_mode) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve selection I/O mode");

    
    if (H5P_get(dx_plist, H5D_XFER_NO_SELECTION_IO_CAUSE_NAME, &H5CX_def_dxpl_cache.no_selection_io_cause) <
        0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve cause for no selection I/O");

    
    if (H5P_get(dx_plist, H5D_XFER_ACTUAL_SELECTION_IO_MODE_NAME,
                &H5CX_def_dxpl_cache.actual_selection_io_mode) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve actual selection I/O mode");

    
    if (H5P_get(dx_plist, H5D_XFER_MODIFY_WRITE_BUF_NAME, &H5CX_def_dxpl_cache.modify_write_buf) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve modify write buffer property");

    
    memset(&H5CX_def_lcpl_cache, 0, sizeof(H5CX_lcpl_cache_t));

    

    
    if (NULL == (lc_plist = (H5P_genplist_t *)H5I_object(H5P_LINK_CREATE_DEFAULT)))
        HGOTO_ERROR(H5E_CONTEXT, H5E_BADTYPE, FAIL, "not a link creation property list");

    
    if (H5P_get(lc_plist, H5P_STRCRT_CHAR_ENCODING_NAME, &H5CX_def_lcpl_cache.encoding) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve link name encoding");

    
    if (H5P_get(lc_plist, H5L_CRT_INTERMEDIATE_GROUP_NAME, &H5CX_def_lcpl_cache.intermediate_group) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve intermediate group creation flag");

    
    memset(&H5CX_def_lapl_cache, 0, sizeof(H5CX_lapl_cache_t));

    

    
    if (NULL == (la_plist = (H5P_genplist_t *)H5I_object(H5P_LINK_ACCESS_DEFAULT)))
        HGOTO_ERROR(H5E_CONTEXT, H5E_BADTYPE, FAIL, "not a link access property list");

    
    if (H5P_get(la_plist, H5L_ACS_NLINKS_NAME, &H5CX_def_lapl_cache.nlinks) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve number of soft / UD links to traverse");

    
    memset(&H5CX_def_dcpl_cache, 0, sizeof(H5CX_dcpl_cache_t));

    

    
    if (NULL == (dc_plist = (H5P_genplist_t *)H5I_object(H5P_DATASET_CREATE_DEFAULT)))
        HGOTO_ERROR(H5E_CONTEXT, H5E_BADTYPE, FAIL, "not a dataset create property list");

    
    if (H5P_get(dc_plist, H5D_CRT_MIN_DSET_HDR_SIZE_NAME, &H5CX_def_dcpl_cache.do_min_dset_ohdr) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve dataset minimize flag");

    
    if (H5P_get(dc_plist, H5O_CRT_OHDR_FLAGS_NAME, &H5CX_def_dcpl_cache.ohdr_flags) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve object header flags");

    
    memset(&H5CX_def_dapl_cache, 0, sizeof(H5CX_dapl_cache_t));

    

    
    if (NULL == (da_plist = (H5P_genplist_t *)H5I_object(H5P_DATASET_ACCESS_DEFAULT)))
        HGOTO_ERROR(H5E_CONTEXT, H5E_BADTYPE, FAIL, "not a dataset create property list");

    
    if (H5P_peek(da_plist, H5D_ACS_EFILE_PREFIX_NAME, &H5CX_def_dapl_cache.extfile_prefix) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve prefix for external file");

    
    if (H5P_peek(da_plist, H5D_ACS_VDS_PREFIX_NAME, &H5CX_def_dapl_cache.vds_prefix) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve prefix for VDS");

    
    memset(&H5CX_def_fapl_cache, 0, sizeof(H5CX_fapl_cache_t));

    

    
    if (NULL == (fa_plist = (H5P_genplist_t *)H5I_object(H5P_FILE_ACCESS_DEFAULT)))
        HGOTO_ERROR(H5E_CONTEXT, H5E_BADTYPE, FAIL, "not a dataset create property list");

    
    if (H5P_get(fa_plist, H5F_ACS_LIBVER_LOW_BOUND_NAME, &H5CX_def_fapl_cache.low_bound) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve dataset minimize flag");

    if (H5P_get(fa_plist, H5F_ACS_LIBVER_HIGH_BOUND_NAME, &H5CX_def_fapl_cache.high_bound) < 0)
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve dataset minimize flag");

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

int
H5CX_term_package(void)
{
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    if (H5_PKG_INIT_VAR) {
        H5CX_node_t **head = NULL; 

        
        head = H5CX_get_my_context();
        assert(head);

        
        *head = NULL;

        H5_PKG_INIT_VAR = false;
    } 

    FUNC_LEAVE_NOAPI(0)
} 

bool
H5CX_pushed(void)
{
    H5CX_node_t **head      = NULL;  
    bool          is_pushed = false; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    head = H5CX_get_my_context(); 
    assert(head);

    
    is_pushed = (*head != NULL);

    FUNC_LEAVE_NOAPI(is_pushed)
}

herr_t
H5CX_push(H5CX_node_t *cnode)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(cnode);
    head = H5CX_get_my_context(); 
    assert(head);

    
    cnode->ctx.dxpl_id = H5P_DATASET_XFER_DEFAULT;
    cnode->ctx.dcpl_id = H5P_DATASET_CREATE_DEFAULT;
    cnode->ctx.dapl_id = H5P_DATASET_ACCESS_DEFAULT;
    cnode->ctx.lcpl_id = H5P_LINK_CREATE_DEFAULT;
    cnode->ctx.lapl_id = H5P_LINK_ACCESS_DEFAULT;
    cnode->ctx.fapl_id = H5P_FILE_ACCESS_DEFAULT;
    cnode->ctx.tag     = H5AC__INVALID_TAG;
    cnode->ctx.ring    = H5AC_RING_USER;

#ifdef H5_HAVE_PARALLEL
    cnode->ctx.btype = MPI_BYTE;
    cnode->ctx.ftype = MPI_BYTE;
#endif

    
    cnode->next = *head;
    *head       = cnode;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_retrieve_state(H5CX_state_t **api_state)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(api_state);

    
    if (NULL == (*api_state = H5FL_CALLOC(H5CX_state_t)))
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTALLOC, FAIL, "unable to allocate new API context state");

    
    if (H5P_DATASET_CREATE_DEFAULT != (*head)->ctx.dcpl_id) {
        
        H5CX_RETRIEVE_PLIST(dcpl, FAIL)

        
        if (((*api_state)->dcpl_id = H5P_copy_plist((H5P_genplist_t *)(*head)->ctx.dcpl, false)) < 0)
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTCOPY, FAIL, "can't copy property list");
    } 
    else
        (*api_state)->dcpl_id = H5P_DATASET_CREATE_DEFAULT;

    
    if (H5P_DATASET_XFER_DEFAULT != (*head)->ctx.dxpl_id) {
        
        H5CX_RETRIEVE_PLIST(dxpl, FAIL)

        
        if (((*api_state)->dxpl_id = H5P_copy_plist((H5P_genplist_t *)(*head)->ctx.dxpl, false)) < 0)
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTCOPY, FAIL, "can't copy property list");
    } 
    else
        (*api_state)->dxpl_id = H5P_DATASET_XFER_DEFAULT;

    
    if (H5P_LINK_ACCESS_DEFAULT != (*head)->ctx.lapl_id) {
        
        H5CX_RETRIEVE_PLIST(lapl, FAIL)

        
        if (((*api_state)->lapl_id = H5P_copy_plist((H5P_genplist_t *)(*head)->ctx.lapl, false)) < 0)
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTCOPY, FAIL, "can't copy property list");
    } 
    else
        (*api_state)->lapl_id = H5P_LINK_ACCESS_DEFAULT;

    
    if (H5P_LINK_CREATE_DEFAULT != (*head)->ctx.lcpl_id) {
        
        H5CX_RETRIEVE_PLIST(lcpl, FAIL)

        
        if (((*api_state)->lcpl_id = H5P_copy_plist((H5P_genplist_t *)(*head)->ctx.lcpl, false)) < 0)
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTCOPY, FAIL, "can't copy property list");
    } 
    else
        (*api_state)->lcpl_id = H5P_LINK_CREATE_DEFAULT;

    
    (*api_state)->vol_wrap_ctx = (*head)->ctx.vol_wrap_ctx;
    if (NULL != (*api_state)->vol_wrap_ctx) {
        assert((*head)->ctx.vol_wrap_ctx_valid);
        if (H5VL_inc_vol_wrapper((*api_state)->vol_wrap_ctx) < 0)
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTINC, FAIL, "can't increment refcount on VOL wrapping context");
    } 

    
    if ((*head)->ctx.vol_connector_prop_valid && (*head)->ctx.vol_connector_prop.connector) {
        
        H5MM_memcpy(&(*api_state)->vol_connector_prop, &(*head)->ctx.vol_connector_prop,
                    sizeof(H5VL_connector_prop_t));

        
        if ((*api_state)->vol_connector_prop.connector) {
            
            if ((*api_state)->vol_connector_prop.connector_info) {
                void *new_connector_info = NULL; 

                
                if (H5VL_copy_connector_info((*api_state)->vol_connector_prop.connector, &new_connector_info,
                                             (*api_state)->vol_connector_prop.connector_info) < 0)
                    HGOTO_ERROR(H5E_CONTEXT, H5E_CANTCOPY, FAIL, "connector info copy failed");
                (*api_state)->vol_connector_prop.connector_info = new_connector_info;
            } 

            
            if (H5VL_conn_inc_rc((*api_state)->vol_connector_prop.connector) < 0)
                HGOTO_ERROR(H5E_CONTEXT, H5E_CANTINC, FAIL, "incrementing VOL connector refcount failed");
        } 
    }     

#ifdef H5_HAVE_PARALLEL
    
    (*api_state)->coll_metadata_read = (*head)->ctx.coll_metadata_read;
#endif 

done:
    
    if (ret_value < 0) {
        if (*api_state) {
            
            if (H5CX_free_state(*api_state) < 0)
                HDONE_ERROR(H5E_CONTEXT, H5E_CANTRELEASE, FAIL, "unable to release API state");
            *api_state = NULL;
        } 
    }     

    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_restore_state(const H5CX_state_t *api_state)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(api_state);

    
    (*head)->ctx.dcpl_id = api_state->dcpl_id;
    (*head)->ctx.dcpl    = NULL;

    
    (*head)->ctx.dxpl_id = api_state->dxpl_id;
    (*head)->ctx.dxpl    = NULL;

    
    (*head)->ctx.lapl_id = api_state->lapl_id;
    (*head)->ctx.lapl    = NULL;

    
    (*head)->ctx.lcpl_id = api_state->lcpl_id;
    (*head)->ctx.lcpl    = NULL;

    
    (*head)->ctx.vol_wrap_ctx = api_state->vol_wrap_ctx;
    if (NULL != (*head)->ctx.vol_wrap_ctx)
        (*head)->ctx.vol_wrap_ctx_valid = true;

    
    if (api_state->vol_connector_prop.connector) {
        H5MM_memcpy(&(*head)->ctx.vol_connector_prop, &api_state->vol_connector_prop,
                    sizeof(H5VL_connector_prop_t));
        (*head)->ctx.vol_connector_prop_valid = true;
    } 

#ifdef H5_HAVE_PARALLEL
    
    (*head)->ctx.coll_metadata_read = api_state->coll_metadata_read;
#endif 

    FUNC_LEAVE_NOAPI(SUCCEED)
} 

herr_t
H5CX_free_state(H5CX_state_t *api_state)
{
    herr_t ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(api_state);

    
    if (0 != api_state->dcpl_id && H5P_DATASET_CREATE_DEFAULT != api_state->dcpl_id)
        if (H5I_dec_ref(api_state->dcpl_id) < 0)
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTDEC, FAIL, "can't decrement refcount on DCPL");

    
    if (0 != api_state->dxpl_id && H5P_DATASET_XFER_DEFAULT != api_state->dxpl_id)
        if (H5I_dec_ref(api_state->dxpl_id) < 0)
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTDEC, FAIL, "can't decrement refcount on DXPL");

    
    if (0 != api_state->lapl_id && H5P_LINK_ACCESS_DEFAULT != api_state->lapl_id)
        if (H5I_dec_ref(api_state->lapl_id) < 0)
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTDEC, FAIL, "can't decrement refcount on LAPL");

    
    if (0 != api_state->lcpl_id && H5P_LINK_CREATE_DEFAULT != api_state->lcpl_id)
        if (H5I_dec_ref(api_state->lcpl_id) < 0)
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTDEC, FAIL, "can't decrement refcount on LCPL");

    
    if (api_state->vol_wrap_ctx)
        if (H5VL_dec_vol_wrapper(api_state->vol_wrap_ctx) < 0)
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTDEC, FAIL, "can't decrement refcount on VOL wrapping context");

    
    if (api_state->vol_connector_prop.connector) {
        
        if (api_state->vol_connector_prop.connector_info)
            if (H5VL_free_connector_info(api_state->vol_connector_prop.connector,
                                         api_state->vol_connector_prop.connector_info) < 0)
                HGOTO_ERROR(H5E_CONTEXT, H5E_CANTRELEASE, FAIL,
                            "unable to release VOL connector info object");

        
        if (H5VL_conn_dec_rc(api_state->vol_connector_prop.connector) < 0)
            HDONE_ERROR(H5E_CONTEXT, H5E_CANTDEC, FAIL, "can't close VOL connector");
    } 

    
    api_state = H5FL_FREE(H5CX_state_t, api_state);

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5CX_is_def_dxpl(void)
{
    H5CX_node_t **head        = NULL;  
    bool          is_def_dxpl = false; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    is_def_dxpl = ((*head)->ctx.dxpl_id == H5P_DATASET_XFER_DEFAULT);

    FUNC_LEAVE_NOAPI(is_def_dxpl)
} 

void
H5CX_set_dxpl(hid_t dxpl_id)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    (*head)->ctx.dxpl_id = dxpl_id;

    FUNC_LEAVE_NOAPI_VOID
} 

void
H5CX_set_dcpl(hid_t dcpl_id)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    (*head)->ctx.dcpl_id = dcpl_id;

    FUNC_LEAVE_NOAPI_VOID
} 

herr_t
H5CX_set_libver_bounds(H5F_t *f)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    (*head)->ctx.low_bound  = (f == NULL) ? H5F_LIBVER_LATEST : H5F_LOW_BOUND(f);
    (*head)->ctx.high_bound = (f == NULL) ? H5F_LIBVER_LATEST : H5F_HIGH_BOUND(f);

    
    (*head)->ctx.low_bound_valid  = true;
    (*head)->ctx.high_bound_valid = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5CX_set_lcpl(hid_t lcpl_id)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    (*head)->ctx.lcpl_id = lcpl_id;

    FUNC_LEAVE_NOAPI_VOID
} 

void
H5CX_set_lapl(hid_t lapl_id)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    (*head)->ctx.lapl_id = lapl_id;

    FUNC_LEAVE_NOAPI_VOID
} 

herr_t
H5CX_set_apl(hid_t *acspl_id, const H5P_libclass_t *libclass,
             hid_t
#ifndef H5_HAVE_PARALLEL
                 H5_ATTR_UNUSED
#endif 
                     loc_id,
             bool
#ifndef H5_HAVE_PARALLEL
                 H5_ATTR_UNUSED
#endif 
                     is_collective)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(acspl_id);
    assert(libclass);
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    if (H5P_DEFAULT == *acspl_id)
        *acspl_id = *libclass->def_plist_id;
    else {
        htri_t is_lapl; 
        htri_t is_dapl; 
        htri_t is_fapl; 

#ifdef H5CX_DEBUG
        
        if (true != H5P_isa_class(*acspl_id, *libclass->class_id))
            HGOTO_ERROR(H5E_CONTEXT, H5E_BADTYPE, FAIL, "not the required access property list");
#endif 

        
        if ((is_lapl = H5P_class_isa(*libclass->pclass, *H5P_CLS_LACC->pclass)) < 0)
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "can't check for link access class");
        else if (is_lapl)
            (*head)->ctx.lapl_id = *acspl_id;

        
        if ((is_dapl = H5P_class_isa(*libclass->pclass, *H5P_CLS_DACC->pclass)) < 0)
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "can't check for dataset access class");
        else if (is_dapl)
            (*head)->ctx.dapl_id = *acspl_id;

        
        if ((is_fapl = H5P_class_isa(*libclass->pclass, *H5P_CLS_FACC->pclass)) < 0)
            HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "can't check for file access class");
        else if (is_fapl)
            (*head)->ctx.fapl_id = *acspl_id;

#ifdef H5_HAVE_PARALLEL
        
        if (!is_collective) {
            H5P_genplist_t         *plist;        
            H5P_coll_md_read_flag_t md_coll_read; 

            
            if (NULL == (plist = (H5P_genplist_t *)H5I_object(*acspl_id)))
                HGOTO_ERROR(H5E_CONTEXT, H5E_BADID, FAIL, "can't find object for ID");

            
            if (H5P_peek(plist, H5_COLL_MD_READ_FLAG_NAME, &md_coll_read) < 0)
                HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "can't get core collective metadata read flag");

            
            if (H5P_USER_TRUE == md_coll_read)
                is_collective = true;
        } 
#endif    
    }     

#ifdef H5_HAVE_PARALLEL
    
    if (is_collective) {
        
        (*head)->ctx.coll_metadata_read = true;

        
        if (H5_coll_api_sanity_check_g) {
            MPI_Comm mpi_comm; 

            
            if (H5F_mpi_retrieve_comm(loc_id, *acspl_id, &mpi_comm) < 0)
                HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get MPI communicator");

            
            if (mpi_comm != MPI_COMM_NULL)
                MPI_Barrier(mpi_comm);
        } 
    }     
#endif    

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_set_loc(hid_t
#ifndef H5_HAVE_PARALLEL
                 H5_ATTR_UNUSED
#endif 
                     loc_id)
{
#ifdef H5_HAVE_PARALLEL
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    (*head)->ctx.coll_metadata_read = true;

    
    if (H5_coll_api_sanity_check_g) {
        MPI_Comm mpi_comm; 

        
        if (H5F_mpi_retrieve_comm(loc_id, H5P_DEFAULT, &mpi_comm) < 0)
            HGOTO_ERROR(H5E_FILE, H5E_CANTGET, FAIL, "can't get MPI communicator");

        
        if (mpi_comm != MPI_COMM_NULL)
            MPI_Barrier(mpi_comm);
    } 

done:
    FUNC_LEAVE_NOAPI(ret_value)
#else  
    FUNC_ENTER_NOAPI_NOINIT_NOERR

    FUNC_LEAVE_NOAPI(SUCCEED)
#endif 
} 

herr_t
H5CX_set_vol_wrap_ctx(void *vol_wrap_ctx)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    (*head)->ctx.vol_wrap_ctx = vol_wrap_ctx;

    
    (*head)->ctx.vol_wrap_ctx_valid = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_set_vol_connector_prop(const H5VL_connector_prop_t *vol_connector_prop)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    H5MM_memcpy(&(*head)->ctx.vol_connector_prop, vol_connector_prop, sizeof(H5VL_connector_prop_t));

    
    (*head)->ctx.vol_connector_prop_valid = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

hid_t
H5CX_get_dxpl(void)
{
    H5CX_node_t **head    = NULL;            
    hid_t         dxpl_id = H5I_INVALID_HID; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    dxpl_id = (*head)->ctx.dxpl_id;

    FUNC_LEAVE_NOAPI(dxpl_id)
} 

hid_t
H5CX_get_lapl(void)
{
    H5CX_node_t **head    = NULL;            
    hid_t         lapl_id = H5I_INVALID_HID; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    lapl_id = (*head)->ctx.lapl_id;

    FUNC_LEAVE_NOAPI(lapl_id)
} 

herr_t
H5CX_get_vol_wrap_ctx(void **vol_wrap_ctx)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vol_wrap_ctx);
    head = H5CX_get_my_context(); 

    
    if (!head)
        HGOTO_ERROR(H5E_CONTEXT, H5E_UNINITIALIZED, FAIL, "the API context isn't available");

    if (!(*head))
        HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "unable to get the current API context");

    
    if ((*head)->ctx.vol_wrap_ctx_valid)
        
        *vol_wrap_ctx = (*head)->ctx.vol_wrap_ctx;
    else
        *vol_wrap_ctx = NULL;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_vol_connector_prop(H5VL_connector_prop_t *vol_connector_prop)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vol_connector_prop);
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    if ((*head)->ctx.vol_connector_prop_valid)
        
        H5MM_memcpy(vol_connector_prop, &(*head)->ctx.vol_connector_prop, sizeof(H5VL_connector_prop_t));
    else
        memset(vol_connector_prop, 0, sizeof(H5VL_connector_prop_t));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

haddr_t
H5CX_get_tag(void)
{
    H5CX_node_t **head = NULL;        
    haddr_t       tag  = HADDR_UNDEF; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    tag = (*head)->ctx.tag;

    FUNC_LEAVE_NOAPI(tag)
} 

H5AC_ring_t
H5CX_get_ring(void)
{
    H5CX_node_t **head = NULL;          
    H5AC_ring_t   ring = H5AC_RING_INV; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    ring = (*head)->ctx.ring;

    FUNC_LEAVE_NOAPI(ring)
} 

#ifdef H5_HAVE_PARALLEL

bool
H5CX_get_coll_metadata_read(void)
{
    H5CX_node_t **head         = NULL; 
    bool          coll_md_read = false;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    coll_md_read = (*head)->ctx.coll_metadata_read;

    FUNC_LEAVE_NOAPI(coll_md_read)
} 

herr_t
H5CX_get_mpi_coll_datatypes(MPI_Datatype *btype, MPI_Datatype *ftype)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(btype);
    assert(ftype);
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    *btype = (*head)->ctx.btype;
    *ftype = (*head)->ctx.ftype;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

bool
H5CX_get_mpi_file_flushing(void)
{
    H5CX_node_t **head     = NULL; 
    bool          flushing = false;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    flushing = (*head)->ctx.mpi_file_flushing;

    FUNC_LEAVE_NOAPI(flushing)
} 

bool
H5CX_get_mpio_rank0_bcast(void)
{
    H5CX_node_t **head           = NULL; 
    bool          do_rank0_bcast = false;

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    do_rank0_bcast = (*head)->ctx.rank0_bcast;

    FUNC_LEAVE_NOAPI(do_rank0_bcast)
} 
#endif 

herr_t
H5CX_get_btree_split_ratios(double split_ratio[3])
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(split_ratio);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_BTREE_SPLIT_RATIO_NAME,
                             btree_split_ratio)

    
    H5MM_memcpy(split_ratio, &(*head)->ctx.btree_split_ratio, sizeof((*head)->ctx.btree_split_ratio));

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_max_temp_buf(size_t *max_temp_buf)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(max_temp_buf);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_MAX_TEMP_BUF_NAME, max_temp_buf)

    
    *max_temp_buf = (*head)->ctx.max_temp_buf;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_tconv_buf(void **tconv_buf)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(tconv_buf);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_TCONV_BUF_NAME, tconv_buf)

    
    *tconv_buf = (*head)->ctx.tconv_buf;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_bkgr_buf(void **bkgr_buf)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(bkgr_buf);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_BKGR_BUF_NAME, bkgr_buf)

    
    *bkgr_buf = (*head)->ctx.bkgr_buf;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_bkgr_buf_type(H5T_bkg_t *bkgr_buf_type)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(bkgr_buf_type);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_BKGR_BUF_TYPE_NAME, bkgr_buf_type)

    
    *bkgr_buf_type = (*head)->ctx.bkgr_buf_type;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_vec_size(size_t *vec_size)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vec_size);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_HYPER_VECTOR_SIZE_NAME, vec_size)

    
    *vec_size = (*head)->ctx.vec_size;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5_HAVE_PARALLEL

herr_t
H5CX_get_io_xfer_mode(H5FD_mpio_xfer_t *io_xfer_mode)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(io_xfer_mode);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_IO_XFER_MODE_NAME, io_xfer_mode)

    
    *io_xfer_mode = (*head)->ctx.io_xfer_mode;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_mpio_coll_opt(H5FD_mpio_collective_opt_t *mpio_coll_opt)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(mpio_coll_opt);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_MPIO_COLLECTIVE_OPT_NAME, mpio_coll_opt)

    
    *mpio_coll_opt = (*head)->ctx.mpio_coll_opt;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_mpio_local_no_coll_cause(uint32_t *mpio_local_no_coll_cause)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(mpio_local_no_coll_cause);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID_SET(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_MPIO_LOCAL_NO_COLLECTIVE_CAUSE_NAME,
                                 mpio_local_no_coll_cause)

    
    *mpio_local_no_coll_cause = (*head)->ctx.mpio_local_no_coll_cause;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_mpio_global_no_coll_cause(uint32_t *mpio_global_no_coll_cause)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(mpio_global_no_coll_cause);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID_SET(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_MPIO_GLOBAL_NO_COLLECTIVE_CAUSE_NAME,
                                 mpio_global_no_coll_cause)

    
    *mpio_global_no_coll_cause = (*head)->ctx.mpio_global_no_coll_cause;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_mpio_chunk_opt_mode(H5FD_mpio_chunk_opt_t *mpio_chunk_opt_mode)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(mpio_chunk_opt_mode);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_MPIO_CHUNK_OPT_HARD_NAME,
                             mpio_chunk_opt_mode)

    
    *mpio_chunk_opt_mode = (*head)->ctx.mpio_chunk_opt_mode;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_mpio_chunk_opt_num(unsigned *mpio_chunk_opt_num)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(mpio_chunk_opt_num);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_MPIO_CHUNK_OPT_NUM_NAME,
                             mpio_chunk_opt_num)

    
    *mpio_chunk_opt_num = (*head)->ctx.mpio_chunk_opt_num;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_mpio_chunk_opt_ratio(unsigned *mpio_chunk_opt_ratio)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(mpio_chunk_opt_ratio);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_MPIO_CHUNK_OPT_RATIO_NAME,
                             mpio_chunk_opt_ratio)

    
    *mpio_chunk_opt_ratio = (*head)->ctx.mpio_chunk_opt_ratio;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 

herr_t
H5CX_get_err_detect(H5Z_EDC_t *err_detect)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(err_detect);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_EDC_NAME, err_detect)

    
    *err_detect = (*head)->ctx.err_detect;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_filter_cb(H5Z_cb_t *filter_cb)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(filter_cb);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_FILTER_CB_NAME, filter_cb)

    
    *filter_cb = (*head)->ctx.filter_cb;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_data_transform(H5Z_data_xform_t **data_transform)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(data_transform);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    

    
    if (!(*head)->ctx.data_transform_valid) {
        
        if ((*head)->ctx.dxpl_id == H5P_DATASET_XFER_DEFAULT)
            (*head)->ctx.data_transform = H5CX_def_dxpl_cache.data_transform;
        else {
            
            if (NULL == (*head)->ctx.dxpl)
                
                if (NULL == ((*head)->ctx.dxpl = (H5P_genplist_t *)H5I_object((*head)->ctx.dxpl_id)))
                    HGOTO_ERROR(H5E_CONTEXT, H5E_BADTYPE, FAIL,
                                "can't get default dataset transfer property list");

            
            
            if (H5P_peek((*head)->ctx.dxpl, H5D_XFER_XFORM_NAME, &(*head)->ctx.data_transform) < 0)
                HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve data transform info");
        } 

        
        (*head)->ctx.data_transform_valid = true;
    } 

    
    *data_transform = (*head)->ctx.data_transform;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_vlen_alloc_info(H5T_vlen_alloc_info_t *vl_alloc_info)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vl_alloc_info);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    
    if (!(*head)->ctx.vl_alloc_info_valid) {
        
        if ((*head)->ctx.dxpl_id == H5P_DATASET_XFER_DEFAULT)
            (*head)->ctx.vl_alloc_info = H5CX_def_dxpl_cache.vl_alloc_info;
        else {
            
            if (NULL == (*head)->ctx.dxpl)
                
                if (NULL == ((*head)->ctx.dxpl = (H5P_genplist_t *)H5I_object((*head)->ctx.dxpl_id)))
                    HGOTO_ERROR(H5E_CONTEXT, H5E_BADTYPE, FAIL,
                                "can't get default dataset transfer property list");

            
            if (H5P_get((*head)->ctx.dxpl, H5D_XFER_VLEN_ALLOC_NAME, &(*head)->ctx.vl_alloc_info.alloc_func) <
                0)
                HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve VL datatype alloc info");
            if (H5P_get((*head)->ctx.dxpl, H5D_XFER_VLEN_ALLOC_INFO_NAME,
                        &(*head)->ctx.vl_alloc_info.alloc_info) < 0)
                HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve VL datatype alloc info");
            if (H5P_get((*head)->ctx.dxpl, H5D_XFER_VLEN_FREE_NAME, &(*head)->ctx.vl_alloc_info.free_func) <
                0)
                HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve VL datatype alloc info");
            if (H5P_get((*head)->ctx.dxpl, H5D_XFER_VLEN_FREE_INFO_NAME,
                        &(*head)->ctx.vl_alloc_info.free_info) < 0)
                HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve VL datatype alloc info");
        } 

        
        (*head)->ctx.vl_alloc_info_valid = true;
    } 

    
    *vl_alloc_info = (*head)->ctx.vl_alloc_info;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_dt_conv_cb(H5T_conv_cb_t *dt_conv_cb)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(dt_conv_cb);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_CONV_CB_NAME, dt_conv_cb)

    
    *dt_conv_cb = (*head)->ctx.dt_conv_cb;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_selection_io_mode(H5D_selection_io_mode_t *selection_io_mode)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(selection_io_mode);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_SELECTION_IO_MODE_NAME,
                             selection_io_mode)

    
    *selection_io_mode = (*head)->ctx.selection_io_mode;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_no_selection_io_cause(uint32_t *no_selection_io_cause)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(no_selection_io_cause);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID_SET(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_NO_SELECTION_IO_CAUSE_NAME,
                                 no_selection_io_cause)

    
    *no_selection_io_cause = (*head)->ctx.no_selection_io_cause;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_actual_selection_io_mode(uint32_t *actual_selection_io_mode)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(actual_selection_io_mode);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    
    if ((*head)->ctx.dxpl_id != H5P_DATASET_XFER_DEFAULT && !(*head)->ctx.actual_selection_io_mode_set &&
        !(*head)->ctx.actual_selection_io_mode_valid) {
        (*head)->ctx.actual_selection_io_mode     = H5CX_def_dxpl_cache.actual_selection_io_mode;
        (*head)->ctx.actual_selection_io_mode_set = true;
    }
    H5CX_RETRIEVE_PROP_VALID_SET(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_ACTUAL_SELECTION_IO_MODE_NAME,
                                 actual_selection_io_mode)

    
    *actual_selection_io_mode = (*head)->ctx.actual_selection_io_mode;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_modify_write_buf(bool *modify_write_buf)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(modify_write_buf);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(dxpl, H5P_DATASET_XFER_DEFAULT, H5D_XFER_MODIFY_WRITE_BUF_NAME, modify_write_buf)

    
    *modify_write_buf = (*head)->ctx.modify_write_buf;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_encoding(H5T_cset_t *encoding)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(encoding);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.lcpl_id);

    H5CX_RETRIEVE_PROP_VALID(lcpl, H5P_LINK_CREATE_DEFAULT, H5P_STRCRT_CHAR_ENCODING_NAME, encoding)

    
    *encoding = (*head)->ctx.encoding;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_intermediate_group(unsigned *crt_intermed_group)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(crt_intermed_group);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.lcpl_id);

    H5CX_RETRIEVE_PROP_VALID(lcpl, H5P_LINK_CREATE_DEFAULT, H5L_CRT_INTERMEDIATE_GROUP_NAME,
                             intermediate_group)

    
    *crt_intermed_group = (*head)->ctx.intermediate_group;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_nlinks(size_t *nlinks)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(nlinks);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dxpl_id);

    H5CX_RETRIEVE_PROP_VALID(lapl, H5P_LINK_ACCESS_DEFAULT, H5L_ACS_NLINKS_NAME, nlinks)

    
    *nlinks = (*head)->ctx.nlinks;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_libver_bounds(H5F_libver_t *low_bound, H5F_libver_t *high_bound)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(low_bound);
    assert(high_bound);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.fapl_id);

    H5CX_RETRIEVE_PROP_VALID(fapl, H5P_FILE_ACCESS_DEFAULT, H5F_ACS_LIBVER_LOW_BOUND_NAME, low_bound)
    H5CX_RETRIEVE_PROP_VALID(fapl, H5P_FILE_ACCESS_DEFAULT, H5F_ACS_LIBVER_HIGH_BOUND_NAME, high_bound)

    
    *low_bound  = (*head)->ctx.low_bound;
    *high_bound = (*head)->ctx.high_bound;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_dset_min_ohdr_flag(bool *dset_min_ohdr_flag)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(dset_min_ohdr_flag);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dcpl_id);

    H5CX_RETRIEVE_PROP_VALID(dcpl, H5P_DATASET_CREATE_DEFAULT, H5D_CRT_MIN_DSET_HDR_SIZE_NAME,
                             do_min_dset_ohdr)

    
    *dset_min_ohdr_flag = (*head)->ctx.do_min_dset_ohdr;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_ext_file_prefix(const char **extfile_prefix)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(extfile_prefix);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dapl_id);

    
    if (!(*head)->ctx.extfile_prefix_valid) {
        
        if ((*head)->ctx.dapl_id == H5P_DATASET_ACCESS_DEFAULT)
            (*head)->ctx.extfile_prefix = H5CX_def_dapl_cache.extfile_prefix;
        else {
            
            if (NULL == (*head)->ctx.dapl)
                
                if (NULL == ((*head)->ctx.dapl = (H5P_genplist_t *)H5I_object((*head)->ctx.dapl_id)))
                    HGOTO_ERROR(H5E_CONTEXT, H5E_BADTYPE, FAIL,
                                "can't get default dataset access property list");

            
            
            if (H5P_peek((*head)->ctx.dapl, H5D_ACS_EFILE_PREFIX_NAME, &(*head)->ctx.extfile_prefix) < 0)
                HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve external file prefix");
        } 

        
        (*head)->ctx.extfile_prefix_valid = true;
    } 

    
    *extfile_prefix = (*head)->ctx.extfile_prefix;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_get_vds_prefix(const char **vds_prefix)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(vds_prefix);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dapl_id);

    
    if (!(*head)->ctx.vds_prefix_valid) {
        
        if ((*head)->ctx.dapl_id == H5P_DATASET_ACCESS_DEFAULT)
            (*head)->ctx.vds_prefix = H5CX_def_dapl_cache.vds_prefix;
        else {
            
            if (NULL == (*head)->ctx.dapl)
                
                if (NULL == ((*head)->ctx.dapl = (H5P_genplist_t *)H5I_object((*head)->ctx.dapl_id)))
                    HGOTO_ERROR(H5E_CONTEXT, H5E_BADTYPE, FAIL,
                                "can't get default dataset access property list");

            
            
            if (H5P_peek((*head)->ctx.dapl, H5D_ACS_VDS_PREFIX_NAME, &(*head)->ctx.vds_prefix) < 0)
                HGOTO_ERROR(H5E_CONTEXT, H5E_CANTGET, FAIL, "Can't retrieve VDS prefix");
        } 

        
        (*head)->ctx.vds_prefix_valid = true;
    } 

    
    *vds_prefix = (*head)->ctx.vds_prefix;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5CX_set_tag(haddr_t tag)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    (*head)->ctx.tag = tag;

    FUNC_LEAVE_NOAPI_VOID
} 

void
H5CX_set_ring(H5AC_ring_t ring)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    (*head)->ctx.ring = ring;

    FUNC_LEAVE_NOAPI_VOID
} 

#ifdef H5_HAVE_PARALLEL

void
H5CX_set_coll_metadata_read(bool cmdr)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    (*head)->ctx.coll_metadata_read = cmdr;

    FUNC_LEAVE_NOAPI_VOID
} 

herr_t
H5CX_set_mpi_coll_datatypes(MPI_Datatype btype, MPI_Datatype ftype)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    (*head)->ctx.btype = btype;
    (*head)->ctx.ftype = ftype;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_set_io_xfer_mode(H5FD_mpio_xfer_t io_xfer_mode)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    (*head)->ctx.io_xfer_mode = io_xfer_mode;

    
    (*head)->ctx.io_xfer_mode_valid = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_set_mpio_coll_opt(H5FD_mpio_collective_opt_t mpio_coll_opt)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    (*head)->ctx.mpio_coll_opt = mpio_coll_opt;

    
    (*head)->ctx.mpio_coll_opt_valid = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

void
H5CX_set_mpi_file_flushing(bool flushing)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    (*head)->ctx.mpi_file_flushing = flushing;

    FUNC_LEAVE_NOAPI_VOID
} 

void
H5CX_set_mpio_rank0_bcast(bool rank0_bcast)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    (*head)->ctx.rank0_bcast = rank0_bcast;

    FUNC_LEAVE_NOAPI_VOID
} 
#endif 

herr_t
H5CX_set_vlen_alloc_info(H5MM_allocate_t alloc_func, void *alloc_info, H5MM_free_t free_func, void *free_info)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    (*head)->ctx.vl_alloc_info.alloc_func = alloc_func;
    (*head)->ctx.vl_alloc_info.alloc_info = alloc_info;
    (*head)->ctx.vl_alloc_info.free_func  = free_func;
    (*head)->ctx.vl_alloc_info.free_info  = free_info;

    
    (*head)->ctx.vl_alloc_info_valid = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_set_nlinks(size_t nlinks)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    (*head)->ctx.nlinks = nlinks;

    
    (*head)->ctx.nlinks_valid = true;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

#ifdef H5_HAVE_PARALLEL

void
H5CX_set_mpio_actual_chunk_opt(H5D_mpio_actual_chunk_opt_mode_t mpio_actual_chunk_opt)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(!((*head)->ctx.dxpl_id == H5P_DEFAULT || (*head)->ctx.dxpl_id == H5P_DATASET_XFER_DEFAULT));

    
    (*head)->ctx.mpio_actual_chunk_opt     = mpio_actual_chunk_opt;
    (*head)->ctx.mpio_actual_chunk_opt_set = true;

    FUNC_LEAVE_NOAPI_VOID
} 

void
H5CX_set_mpio_actual_io_mode(H5D_mpio_actual_io_mode_t mpio_actual_io_mode)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(!((*head)->ctx.dxpl_id == H5P_DEFAULT || (*head)->ctx.dxpl_id == H5P_DATASET_XFER_DEFAULT));

    
    (*head)->ctx.mpio_actual_io_mode     = mpio_actual_io_mode;
    (*head)->ctx.mpio_actual_io_mode_set = true;

    FUNC_LEAVE_NOAPI_VOID
} 

void
H5CX_set_mpio_local_no_coll_cause(uint32_t mpio_local_no_coll_cause)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert((*head)->ctx.dxpl_id != H5P_DEFAULT);

    
    if ((*head)->ctx.dxpl_id != H5P_DATASET_XFER_DEFAULT) {
        
        (*head)->ctx.mpio_local_no_coll_cause     = mpio_local_no_coll_cause;
        (*head)->ctx.mpio_local_no_coll_cause_set = true;
    } 

    FUNC_LEAVE_NOAPI_VOID
} 

void
H5CX_set_mpio_global_no_coll_cause(uint32_t mpio_global_no_coll_cause)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert((*head)->ctx.dxpl_id != H5P_DEFAULT);

    
    if ((*head)->ctx.dxpl_id != H5P_DATASET_XFER_DEFAULT) {
        
        (*head)->ctx.mpio_global_no_coll_cause     = mpio_global_no_coll_cause;
        (*head)->ctx.mpio_global_no_coll_cause_set = true;
    } 

    FUNC_LEAVE_NOAPI_VOID
} 

#ifdef H5_HAVE_INSTRUMENTED_LIBRARY

herr_t
H5CX_test_set_mpio_coll_chunk_link_hard(int mpio_coll_chunk_link_hard)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(!((*head)->ctx.dxpl_id == H5P_DEFAULT || (*head)->ctx.dxpl_id == H5P_DATASET_XFER_DEFAULT));

    H5CX_TEST_SET_PROP(H5D_XFER_COLL_CHUNK_LINK_HARD_NAME, mpio_coll_chunk_link_hard)

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_test_set_mpio_coll_chunk_multi_hard(int mpio_coll_chunk_multi_hard)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(!((*head)->ctx.dxpl_id == H5P_DEFAULT || (*head)->ctx.dxpl_id == H5P_DATASET_XFER_DEFAULT));

    H5CX_TEST_SET_PROP(H5D_XFER_COLL_CHUNK_MULTI_HARD_NAME, mpio_coll_chunk_multi_hard)

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_test_set_mpio_coll_chunk_link_num_true(int mpio_coll_chunk_link_num_true)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(!((*head)->ctx.dxpl_id == H5P_DEFAULT || (*head)->ctx.dxpl_id == H5P_DATASET_XFER_DEFAULT));

    H5CX_TEST_SET_PROP(H5D_XFER_COLL_CHUNK_LINK_NUM_TRUE_NAME, mpio_coll_chunk_link_num_true)

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_test_set_mpio_coll_chunk_link_num_false(int mpio_coll_chunk_link_num_false)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(!((*head)->ctx.dxpl_id == H5P_DEFAULT || (*head)->ctx.dxpl_id == H5P_DATASET_XFER_DEFAULT));

    H5CX_TEST_SET_PROP(H5D_XFER_COLL_CHUNK_LINK_NUM_FALSE_NAME, mpio_coll_chunk_link_num_false)

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_test_set_mpio_coll_chunk_multi_ratio_coll(int mpio_coll_chunk_multi_ratio_coll)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(!((*head)->ctx.dxpl_id == H5P_DEFAULT || (*head)->ctx.dxpl_id == H5P_DATASET_XFER_DEFAULT));

    H5CX_TEST_SET_PROP(H5D_XFER_COLL_CHUNK_MULTI_RATIO_COLL_NAME, mpio_coll_chunk_multi_ratio_coll)

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_test_set_mpio_coll_chunk_multi_ratio_ind(int mpio_coll_chunk_multi_ratio_ind)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(!((*head)->ctx.dxpl_id == H5P_DEFAULT || (*head)->ctx.dxpl_id == H5P_DATASET_XFER_DEFAULT));

    H5CX_TEST_SET_PROP(H5D_XFER_COLL_CHUNK_MULTI_RATIO_IND_NAME, mpio_coll_chunk_multi_ratio_ind)

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_test_set_mpio_coll_rank0_bcast(bool mpio_coll_rank0_bcast)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI_NOINIT

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(!((*head)->ctx.dxpl_id == H5P_DEFAULT || (*head)->ctx.dxpl_id == H5P_DATASET_XFER_DEFAULT));

    H5CX_TEST_SET_PROP(H5D_XFER_COLL_RANK0_BCAST_NAME, mpio_coll_rank0_bcast)

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
#endif 
#endif 

void
H5CX_set_no_selection_io_cause(uint32_t no_selection_io_cause)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert((*head)->ctx.dxpl_id != H5P_DEFAULT);

    
    if ((*head)->ctx.dxpl_id != H5P_DATASET_XFER_DEFAULT) {
        
        (*head)->ctx.no_selection_io_cause     = no_selection_io_cause;
        (*head)->ctx.no_selection_io_cause_set = true;
    } 

    FUNC_LEAVE_NOAPI_VOID
} 

void
H5CX_set_actual_selection_io_mode(uint32_t actual_selection_io_mode)
{
    H5CX_node_t **head = NULL; 

    FUNC_ENTER_NOAPI_NOINIT_NOERR

    
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert((*head)->ctx.dxpl_id != H5P_DEFAULT);

    
    if ((*head)->ctx.dxpl_id != H5P_DATASET_XFER_DEFAULT) {
        
        (*head)->ctx.actual_selection_io_mode     = actual_selection_io_mode;
        (*head)->ctx.actual_selection_io_mode_set = true;
    }

    FUNC_LEAVE_NOAPI_VOID
} 

herr_t
H5CX_get_ohdr_flags(uint8_t *ohdr_flags)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    assert(ohdr_flags);
    head = H5CX_get_my_context(); 
    assert(head && *head);
    assert(H5P_DEFAULT != (*head)->ctx.dcpl_id);

    H5CX_RETRIEVE_PROP_VALID(dcpl, H5P_DATASET_CREATE_DEFAULT, H5O_CRT_OHDR_FLAGS_NAME, ohdr_flags)

    
    *ohdr_flags = (*head)->ctx.ohdr_flags;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 

herr_t
H5CX_pop(bool update_dxpl_props)
{
    H5CX_node_t **head      = NULL;    
    herr_t        ret_value = SUCCEED; 

    FUNC_ENTER_NOAPI(FAIL)

    
    head = H5CX_get_my_context(); 
    assert(head && *head);

    
    if (update_dxpl_props) {
        
        if ((*head)->ctx.dxpl_id != H5P_DATASET_XFER_DEFAULT && !(*head)->ctx.actual_selection_io_mode_set) {
            (*head)->ctx.actual_selection_io_mode     = H5CX_def_dxpl_cache.actual_selection_io_mode;
            (*head)->ctx.actual_selection_io_mode_set = true;
        }

        H5CX_SET_PROP(H5D_XFER_NO_SELECTION_IO_CAUSE_NAME, no_selection_io_cause)
        H5CX_SET_PROP(H5D_XFER_ACTUAL_SELECTION_IO_MODE_NAME, actual_selection_io_mode)
#ifdef H5_HAVE_PARALLEL
        H5CX_SET_PROP(H5D_MPIO_ACTUAL_CHUNK_OPT_MODE_NAME, mpio_actual_chunk_opt)
        H5CX_SET_PROP(H5D_MPIO_ACTUAL_IO_MODE_NAME, mpio_actual_io_mode)
        H5CX_SET_PROP(H5D_MPIO_LOCAL_NO_COLLECTIVE_CAUSE_NAME, mpio_local_no_coll_cause)
        H5CX_SET_PROP(H5D_MPIO_GLOBAL_NO_COLLECTIVE_CAUSE_NAME, mpio_global_no_coll_cause)
#ifdef H5_HAVE_INSTRUMENTED_LIBRARY
        H5CX_SET_PROP(H5D_XFER_COLL_CHUNK_LINK_HARD_NAME, mpio_coll_chunk_link_hard)
        H5CX_SET_PROP(H5D_XFER_COLL_CHUNK_MULTI_HARD_NAME, mpio_coll_chunk_multi_hard)
        H5CX_SET_PROP(H5D_XFER_COLL_CHUNK_LINK_NUM_TRUE_NAME, mpio_coll_chunk_link_num_true)
        H5CX_SET_PROP(H5D_XFER_COLL_CHUNK_LINK_NUM_FALSE_NAME, mpio_coll_chunk_link_num_false)
        H5CX_SET_PROP(H5D_XFER_COLL_CHUNK_MULTI_RATIO_COLL_NAME, mpio_coll_chunk_multi_ratio_coll)
        H5CX_SET_PROP(H5D_XFER_COLL_CHUNK_MULTI_RATIO_IND_NAME, mpio_coll_chunk_multi_ratio_ind)
        H5CX_SET_PROP(H5D_XFER_COLL_RANK0_BCAST_NAME, mpio_coll_rank0_bcast)
#endif 
#endif 
    }  

    
    (*head) = (*head)->next;

done:
    FUNC_LEAVE_NOAPI(ret_value)
} 
