% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gkwreg-datasets.R
\docType{data}
\name{CarTask}
\alias{CarTask}
\title{Partition-Primed Probability Judgement Task for Car Dealership}
\format{
A data frame with 155 observations on 3 variables:
\describe{
\item{probability}{numeric. Estimated probability (response variable).}
\item{task}{factor with levels \code{Car} and \code{Salesperson} indicating the
condition/question type.}
\item{NFCCscale}{numeric. Combined score from the Need for Closure (NFC) and
Need for Certainty (NCC) scales, which are strongly correlated.}
}
}
\source{
Taken from Smithson et al. (2011) supplements.
}
\usage{
CarTask
}
\description{
Data from a cognitive experiment examining how partition priming affects
probability judgments in a car dealership context. Participants judged
probabilities under different framing conditions.
}
\details{
All participants in the study were undergraduate students at The Australian
National University, some of whom obtained course credit in first-year Psychology
for their participation.

\strong{Task questions:}
\itemize{
\item \strong{Car condition:} "What is the probability that a customer trades in a coupe?"
\item \strong{Salesperson condition:} "What is the probability that a customer buys a car
from Carlos?" (out of four possible salespersons)
}

The key manipulation is the implicit partition: In the Car condition, there are
multiple car types (binary: coupe vs. not coupe), while in the Salesperson
condition, there are four specific salespersons. Classical findings suggest that
different partition structures lead to different probability estimates even when
the actual probabilities are equivalent.

The NFCC scale (Need for Closure and Certainty) measures individual differences
in tolerance for ambiguity. Higher scores indicate greater need for definitive
answers and discomfort with uncertainty.
}
\examples{
\donttest{
require(gkwreg)
require(gkwdist)

data(CarTask)

# Example 1: Task effects on probability judgments
# Do people judge probabilities differently for car vs. salesperson?
fit_kw <- gkwreg(
  probability ~ task,
  data = CarTask,
  family = "kw"
)
summary(fit_kw)

# Interpretation:
# - Alpha: Task type affects mean probability estimate
#   Salesperson condition (1/4 = 0.25) vs. car type (unclear baseline)

# Example 2: Individual differences model
# Need for Closure/Certainty may moderate probability judgments
fit_kw_nfcc <- gkwreg(
  probability ~ task * NFCCscale |
    task,
  data = CarTask,
  family = "kw"
)
summary(fit_kw_nfcc)

# Interpretation:
# - Interaction: NFCC may have different effects depending on task
#   People high in need for certainty may respond differently to
#   explicit partitions (4 salespersons) vs. implicit partitions (car types)
# - Beta: Precision varies by task type

# Example 3: Exponentiated Kumaraswamy for extreme estimates
# Some participants may give very extreme probability estimates
fit_ekw <- gkwreg(
  probability ~ task * NFCCscale | # alpha
    task | # beta
    task, # lambda: extremity differs by task
  data = CarTask,
  family = "ekw"
)
summary(fit_ekw)

# Interpretation:
# - Lambda varies by task: Salesperson condition (explicit partition)
#   may produce more extreme estimates (closer to 0 or 1)

# Visualization: Probability by task and NFCC
plot(probability ~ NFCCscale,
  data = CarTask,
  col = c("blue", "red")[task], pch = 19,
  xlab = "Need for Closure/Certainty", ylab = "Probability Estimate",
  main = "Car Task: Individual Differences in Probability Judgment"
)
legend("topright",
  legend = levels(CarTask$task),
  col = c("blue", "red"), pch = 19
)

# Distribution comparison
boxplot(probability ~ task,
  data = CarTask,
  xlab = "Task Condition", ylab = "Probability Estimate",
  main = "Partition Priming Effects",
  col = c("lightblue", "lightcoral")
)
abline(h = 0.25, lty = 2, col = "gray")
text(1.5, 0.27, "Uniform (1/4)", col = "gray")
}
}
\references{
Smithson, M., Merkle, E.C., and Verkuilen, J. (2011). Beta Regression Finite
Mixture Models of Polarization and Priming. \emph{Journal of Educational and
Behavioral Statistics}, \strong{36}(6), 804--831.
\doi{10.3102/1076998610396893}

Smithson, M., and Segale, C. (2009). Partition Priming in Judgments of
Imprecise Probabilities. \emph{Journal of Statistical Theory and Practice},
\strong{3}(1), 169--181.
}
\keyword{datasets}
