% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/gerda_covariates.R
\name{add_gerda_covariates}
\alias{add_gerda_covariates}
\title{Add County-Level Covariates to GERDA Election Data}
\usage{
add_gerda_covariates(election_data)
}
\arguments{
\item{election_data}{A data frame containing GERDA election data. Must
contain a column with county or municipal codes (see Details) and
\code{election_year}.}
}
\value{
The input data frame with additional columns for all 20 county-level
covariates. The number of rows remains unchanged (left join).
}
\description{
Convenience function to merge INKAR county-level (Kreis) covariates with
GERDA election data. This is the recommended way to add covariates, as it
automatically uses the correct join keys and prevents common merge errors.

The function works with both county-level and municipal-level election data:
\itemize{
\item \strong{County-level data}: Direct merge using county codes
\item \strong{Municipal-level data}: Automatically extracts county code
from municipal AGS (first 5 digits) and merges
}

\strong{Important}: Covariates are always at the county level. When merging
with municipal data, all municipalities within the same county will receive
identical covariate values.

The function performs a left join, keeping all rows from the election data
and adding covariates where available. This automatically retains only
election years.
}
\details{
\subsection{Required Columns}{

The input data must contain \code{election_year} and one of:
\itemize{
\item \code{county_code}: 5-digit county code (AGS) for county-level data
\item \code{ags}: 8-digit municipal code (AGS) for municipal-level data
}

The function automatically detects which column is present and performs the
appropriate merge. For municipal data, the county code is extracted from the
first 5 digits of the AGS.
}

\subsection{Data Level}{

Covariates are at the county (Kreis) level:
\itemize{
\item \strong{County-level merge}: One-to-one match, each county gets its covariates
\item \strong{Municipal-level merge}: Many-to-one match, all municipalities in the
same county receive identical covariate values
}
}

\subsection{Data Availability}{

Covariates are available from 1995-2022. For GERDA federal elections:
\itemize{
\item Elections 1990, 1994: No covariates (before 1995)
\item Elections 1998-2021: Covariates available
}
}

\subsection{Missing Data}{

Some covariates have missing values. Use \code{gerda_covariates_codebook()}
to check data availability for specific variables.
}
}
\examples{
\dontrun{
library(gerda)
library(dplyr)

# Example 1: County-level election data
county_data <- load_gerda_web("federal_cty_harm") \%>\%
  add_gerda_covariates()

# Check the result
names(county_data) # See new covariate columns
table(county_data$election_year) # Only election years

# Example 2: Municipal-level election data
# Note: All municipalities in the same county will get identical covariates
muni_data <- load_gerda_web("federal_muni_harm_21") \%>\%
  add_gerda_covariates()

# Verify: municipalities in same county have same covariate values
muni_data \%>\%
  group_by(county_code_21, election_year) \%>\%
  summarize(
    n_munis = n(),
    unemp_range = max(unemployment_rate) - min(unemployment_rate)
  )

# Analyze with covariates
county_data \%>\%
  filter(election_year == 2021) \%>\%
  filter(!is.na(unemployment_rate)) \%>\%
  summarize(cor_unemployment_afd = cor(unemployment_rate, afd))
}

}
\seealso{
\itemize{
\item \code{\link{gerda_covariates}} for direct access to the covariate data
\item \code{\link{gerda_covariates_codebook}} for variable descriptions
\item \code{\link{load_gerda_web}} for loading GERDA election data
}
}
