% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/clustering_partition.R
\name{clustering_partition}
\alias{clustering_partition}
\title{Obtain a partition of the spatial domain using the density-based spatial clustering (DBSC) algorithm described in Santafé et al. (2021)}
\usage{
clustering_partition(
  carto,
  ID.area = NULL,
  var = NULL,
  n.cluster = 10,
  min.size = NULL,
  W = NULL,
  l = 1,
  Wk = NULL,
  distance = "euclidean",
  verbose = TRUE
)
}
\arguments{
\item{carto}{object of class \code{SpatialPolygonsDataFrame} or \code{sf}.}

\item{ID.area}{character; name of the variable that contains the IDs of spatial areal units.}

\item{var}{character; name of the variable that contains the data of interest to compute spatial clusters, usually the vector of log-SMR.}

\item{n.cluster}{numeric; value to fix the number of cluster centers in the DBSC algorithm. Default to 10.}

\item{min.size}{numeric (default \code{NULL}); value to fix the minimum size of areas in each spatial partition.}

\item{W}{optional argument with the binary adjacency matrix of the spatial areal units. If \code{NULL} (default), this object is computed from the \code{carto} argument (two areas are considered as neighbours if they share a common border).}

\item{l}{numeric value with the neighbourhood order used to assign areas to each cluster. If \code{k=1} (default), only areas that share a common border are considered.}

\item{Wk}{previously computed binary adjacency matrix of l-order neighbours. If this argument is included (default \code{NULL}), the parameter \code{l} is ignored.}

\item{distance}{the distance measure to be used (default \code{"euclidean"}). See the \code{method} argument of \code{\link[stats]{dist}} function for other options.}

\item{verbose}{logical value (default \code{TRUE}); indicates if the function runs in verbose mode.}
}
\value{
\code{sf} object with the original data and a grouping variable named 'ID.group'.
}
\description{
The function takes an object of class \code{SpatialPolygonsDataFrame} or \code{sf} and defines a spatial partition using the DBSC algorithm described in \insertCite{santafe2021;textual}{bigDM}.
}
\details{
The DBSC algorithm implemented in this function is a new spatial clustering algorithm based on the density clustering algorithm introduced by \insertCite{rodriguez2014clustering;textual}{bigDM} and the posterior modification presented by \insertCite{wang2016automatic;textual}{bigDM}.
This algorithm is able to obtain a single clustering partition of the data by automatically detecting clustering centers and assigning each area to its nearest cluster centroid.
The algorithm has its basis in the assumption that cluster centers are points with high local density and relatively large distance to other points with higher local densities.
See \insertCite{santafe2021;textual}{bigDM} for more details.
}
\examples{
\dontrun{
library(sf)
library(tmap)

## Load the Spain colorectal cancer mortality data ##
data(Carto_SpainMUN)

## Define a spatial partition using the DBSC algorithm ##
Carto_SpainMUN$logSMR <- log(Carto_SpainMUN$obs/Carto_SpainMUN$exp+0.0001)

carto.new <- clustering_partition(carto=Carto_SpainMUN, ID.area="ID", var="logSMR",
                                  n.cluster=20, l=2, min.size=100, verbose=TRUE)
table(carto.new$ID.group)

## Plot of the grouping variable 'ID.group' ##
carto.data <- st_set_geometry(carto.new, NULL)
carto.partition <- aggregate(carto.new[,"geometry"], list(ID.group=carto.data[,"ID.group"]), head)

tmap4 <- packageVersion("tmap") >= "3.99"

if(tmap4){
        tm_shape(carto.new) +
                tm_polygons(fill="ID.group", fill.scale=tm_scale(values="brewer.set3")) +
                tm_shape(carto.partition) +
                tm_borders(col="black", lwd=2) +
                tm_layout(legend.outside=TRUE, legend.frame=FALSE)
}else{
        tm_shape(carto.new) +
                tm_polygons(col="ID.group") +
                tm_shape(carto.partition) +
                tm_borders(col="black", lwd=2) +
                tm_layout(legend.outside=TRUE)
}
}

}
\references{
\insertRef{rodriguez2014clustering}{bigDM}

\insertRef{santafe2021}{bigDM}

\insertRef{wang2016automatic}{bigDM}
}
