% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Islands.R
\name{Islands}
\alias{Islands}
\title{Find islands from distance matrix}
\usage{
Islands(D, threshold, dense = TRUE, smallest = 0)
}
\arguments{
\item{D}{Square matrix or \code{dist} object containing Euclidean distances
between data points.}

\item{threshold}{Elements greater than \code{threshold} distance units will not be
assigned to the same island.}

\item{dense}{Logical; if \code{FALSE}, each island will be named according to the
index of its lowest-indexed member; if \code{TRUE}, each island will be numbered
sequentially from \code{1}, ordered by the index of the lowest-indexed member.}

\item{smallest}{Integer; Islands comprising no more than \code{smallest} elements
will be assigned to island \code{NA}.}
}
\value{
\code{Islands()} returns a vector listing the island to which
each element is assigned.
}
\description{
\code{Islands()} assigns a set of objects to islands, such that all elements
within an island can form a connected graph in which each edge is no longer
than \code{threshold} distance units \insertRef{Silva2021}{TreeDist}.
}
\examples{
library("TreeTools", quietly = TRUE)
# Generate a set of trees
trees <- as.phylo(as.TreeNumber(BalancedTree(16)) + c(-(40:20), 70:105), 16)

# Calculate distances between trees
distances <- ClusteringInfoDist(trees)
summary(distances)

# Assign trees to islands
isle <- Islands(distances, quantile(distances, 0.1))
table(isle)

# Indicate island membership on 2D mapping of tree distances
mapping <- cmdscale(distances, 2)
plot(mapping, col = isle + 1,
     asp = 1, # Preserve aspect ratio - do not distort distances
     ann = FALSE, axes = FALSE, # Don't label axes: dimensions are meaningless)
     pch = 16 # Plotting character: Filled circle
)

# Compare strict consensus with island consensus trees
oPar <- par(mfrow = c(2, 2), mai = rep(0.1, 4))
plot(Consensus(trees), main = "Strict")
plot(Consensus(trees[isle == 1]), edge.col = 2, main = "Island 1")
plot(Consensus(trees[isle == 2]), edge.col = 3, main = "Island 2")
plot(Consensus(trees[isle == 3]), edge.col = 4, main = "Island 3")

# Restore graphical parameters
par(oPar)
}
\references{
\insertAllCited{}
}
\seealso{
Other tree space functions: 
\code{\link{MSTSegments}()},
\code{\link{MapTrees}()},
\code{\link{MappingQuality}()},
\code{\link{SpectralEigens}()},
\code{\link{cluster-statistics}},
\code{\link{median.multiPhylo}()}
}
\author{
\href{https://orcid.org/0000-0001-5660-1727}{Martin R. Smith}
(\href{mailto:martin.smith@durham.ac.uk}{martin.smith@durham.ac.uk})
}
\concept{tree space functions}
