% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/LorMe_pipeline.R
\name{LorMe_pipeline}
\alias{LorMe_pipeline}
\title{One-stop microbial analysis pipeline for LorMe objects}
\usage{
LorMe_pipeline(taxobj, step = "all")
}
\arguments{
\item{taxobj}{A configured \strong{LorMe} object created by
\code{\link{object_config}}.
The object must contain preprocessed taxonomy tables, metadata,
and analysis configurations.}

\item{step}{Character vector specifying which analysis modules to run.
Must be one or more of:
\itemize{
\item \code{"all"} – run the entire pipeline (default)
\item \code{"profile"} – alpha/beta diversity and composition analysis
\item \code{"diff"} – differential abundance (DESeq2, differential barplot)
\item \code{"sub_net"} – treatment-specific subnetworks
\item \code{"all_net"} – combined/co-occurrence network across all samples
}}
}
\value{
A named list containing the results of all requested modules.
Depending on \code{step}, the list may include:
\describe{
\item{\code{alpha_results}}{Alpha diversity tables and plots}
\item{\code{beta_results}}{Beta diversity ordinations (e.g., PCoA)}
\item{\code{composition_results}}{Community composition barplots}
\item{\code{diffbar_result}}{Differential barplot results}
\item{\code{Deseq_results}}{DESeq2 results for each comparison}
\item{\code{Deseq_volcano}}{Volcano plots for differential analysis}
\item{\code{Deseq_manhattan}}{Manhattan plots for differential features}
\item{\code{indicator_results}}{Indicator species results}
\item{\code{indic_volcano}}{Volcano plot of indicator species analysis}
\item{\code{indic_manhattan}}{Manhattan plot of indicator analysis}
\item{\code{sub_network_results}}{Treatment-specific subnetworks}
\item{\code{combine_network_results}}{Global co-occurrence network}
}

Failed modules return as \code{NULL}.
}
\description{
\code{LorMe_pipeline()} performs a complete microbial ecology analysis workflow
for a configured \strong{LorMe} object, including:
community profiling, differential analysis, indicator species analysis,
treatment-specific sub-network construction, and global network construction.

The function is modular and allows users to execute only selected steps
through the \code{step} argument.
}
\details{
The function automatically respects global options set via
\code{\link{LorMe_options}}.
These options control:
\itemize{
\item analysis taxonomic level
\item palettes and plotting parameters
\item DESeq2 parameters
\item network analysis thresholds
}

Internal failures in any analysis sub-module do \strong{not} stop the pipeline:
the corresponding output is returned as \code{NULL}, and the function prints
a summary of failed steps on completion.
}
\examples{
\donttest{
## View current global analysis options
getOption("LorMe")

## Set analysis options
LorMe_options(
  global = list(Analysis_level = "Genus"),
  sub_net = list(threshold = 0.7),
  all_net = list(threshold = 0.7)
)

## Run pipeline (time-consuming)
Two_group_analysis <- LorMe_pipeline(Two_group)

## Access results:
# Alpha diversity
Two_group_analysis$alpha_results$plotlist$Plotobj_Shannon$Boxplot

# Beta diversity
Two_group_analysis$beta_results$PCoA_Plot

# Community composition
Two_group_analysis$composition_results$barplot

# Differential analysis
Two_group_analysis$Deseq_volcano$FC_FDR
Two_group_analysis$Deseq_manhattan$manhattan

# Differential barplot
library(patchwork)
Two_group_analysis$diffbar_result$Barplot |
  Two_group_analysis$diffbar_result$Differenceplot

# Subnetworks
Two_group_analysis$sub_network_results$Treatment_sub_network \%>\%
  network_visual()

# Combined network
Two_group_analysis$combine_network_results \%>\% network_visual()

## Reset to default options
LorMe_defaults()

## Example: three-group comparison with custom options
LorMe_options(
  global = list(
    Analysis_level = "Species",
    compare_list = c("CF_OF", "CF_BF")
  ),
  all_net = list(threshold = 0.95, method = "pearson")
)

Three_group_analysis <- LorMe_pipeline(Three_group)
}
}
