% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/mh_rep_ch.R
\name{mh_rep_ch}
\alias{mh_rep_ch}
\title{Multivariate temporal climate tepresentativeness change analysis}
\usage{
mh_rep_ch(
  polygon,
  col_name,
  present_climate_variables,
  future_climate_variables,
  study_area,
  th = 0.95,
  model,
  year,
  dir_output = file.path(tempdir(), "ClimaRep"),
  save_raw = FALSE
)
}
\arguments{
\item{polygon}{An \code{sf} object containing the defined areas. \strong{Its CRS will be used as the reference system.}}

\item{col_name}{\code{character}. Name of the column in the \code{polygon} object that contains unique identifiers for each polygon.}

\item{present_climate_variables}{A \code{SpatRaster} stack of climate variables representing current conditions.}

\item{future_climate_variables}{A \code{SpatRaster} stack containing the same climate variables as \code{present_climate_variables} but representing future projected conditions.}

\item{study_area}{A single \code{sf} polygon.}

\item{th}{\code{numeric} (0-1). Percentile threshold used to define representativeness. Cells with a Mahalanobis distance below or equal to the \code{th} are classified as representative (default: 0.95).}

\item{model}{\code{character}. Name or identifier of the climate model used (e.g., "MIROC6"). This parameter is used in output filenames and subdirectory names, allowing for better file management.}

\item{year}{\code{character}. Year or period of future climate data (e.g., "2070"). This parameter is used in output filenames and subdirectory names, allowing for better file management.}

\item{dir_output}{\code{character}. Path to the directory where output files will be saved. The function will create subdirectories within this path.}

\item{save_raw}{\code{logical}. If \code{TRUE}, saves the intermediate continuous Mahalanobis distance rasters calculated for each polygon before binary classification. The final binary classification rasters are always saved (default: \code{FALSE}).}
}
\value{
Writes the following outputs to disk within subdirectories of \code{dir_output}:
\itemize{
\item Classification (\code{.tif} ) change rasters: Change category rasters (\code{0} for \strong{Non-representative}, \code{1} for \strong{Retained}, \code{2} for \strong{Lost} and \code{3} for \strong{Novel}) for each input polygon are saved in the \verb{Change/} subdirectory.
\item Visualization (\code{.jpeg}) maps: Image files visualizing the change classification results for each \code{polygon} are saved in the \verb{Charts/} subdirectory.
\item Raw Mahalanobis distance rasters: Optionally, they are saved as \code{.tif} files in the \verb{Mh_Raw_Pre/} and \verb{Mh_Raw_Fut/} subdirectories if \code{save_raw = TRUE}.
}
}
\description{
This function calculates Mahalanobis-based climate representativeness (or forward climate analogs) for input polygon across two time periods (present and future) within a defined area.

The function categorizes cells based on how their climate representativeness changes, labeling them as Retained, Lost, or Novel.

Representativeness is assessed by comparing the multivariate climate conditions of each cell, of the reference climate space (\code{present_climate_variables} and \code{future_climate_variables}), with the climate conditions within each specific input \code{polygon}.
}
\details{
This function extends the approach used in \code{mh_rep} to assess Changes in Climate Representativeness (or forward climate analogs) over time.
While \code{mh_rep()} calculates representativeness in a single scenario, \code{mh_rep_ch()} adapts this by using the mean from the present polygon but a covariance matrix derived from the overall climate space across both present and future periods combined.

Here are the key steps:
\enumerate{
\item Checking CRS: Ensures that \code{future_climate_variables}, \code{climate_variables}, and \code{study_area} have matching CRSs with \code{polygon} by automatically transforming them if needed. The CRS of \code{polygon} will be used as the reference system.
\item Crops and masks the \code{climate_variables} and \code{future_climate_variables} raster to the \code{study_area} to limit all subsequent calculations to the area of interest.
\item Calculate the multivariate covariance matrix using climate data from all cells for both present and present-future time periods combined.
\item For each polygon in the \code{polygon} object:
\itemize{
\item Crop and mask the present climate variables raster (\code{present_climate_variables}) to the boundary of the current polygon.
\item Calculate the multivariate mean using the climate data from the previous step. This defines the climate centroid for the current polygon.
Calculate the Mahalanobis distance for each cell relative to the centroid and the overall present and present-future covariance matrix.
This results in a Mahalanobis distance raster for the present period and another for the future period.
\item Apply the specified threshold (\code{th}) to Mahalanobis distances to determine which cells are considered representative. This threshold is a percentile of the Mahalanobis distances within the current polygon.
\item Classify each cells, for both present and future periods, as Representative = \code{1} (Mahalanobis distance \eqn{\le} \code{th}) or Non-Representative = \code{0} (Mahalanobis distance $>$ \code{th}).
}
\item Compares the binary representativeness of each cell between the present and future periods and determines cells where conditions are:
\itemize{
\item \code{0}: \strong{Non-representative}: Cells that are outside the defined Mahalanobis threshold in both present and future periods.
\item \code{1}: \strong{Retained}: Cells that are within the defined Mahalanobis threshold in both present and future periods. \strong{Representative} if \code{Climarep::mh_rep()} is used
\item \code{2}: \strong{Lost}: Cells that are within the defined Mahalanobis threshold in the present period but outside it in the future period.
\item \code{3}: \strong{Novel}: Cells that are outside the defined Mahalanobis threshold in the present period but within it in the future period.
}
\item Saves the classification raster (\code{.tif}) and generates a corresponding visualization map (\code{.jpeg}) for each polygon. These are saved within the specified output directory (\code{dir_output}).
All files are saved using the \code{model} and \code{year} parameters for better file management.
}

It is important to note that Mahalanobis distance assumes is sensitive to collinearity among variables.
While the covariance matrix accounts for correlations, it is strongly recommended that the climate variables (\code{present_climate_variables}) are not strongly correlated.
Consider performing a collinearity analysis beforehand, perhaps using the \code{vif_filter()} function from this package.
}
\examples{
library(terra)
library(sf)
set.seed(2458)
n_cells <- 100 * 100
r_clim_present <- terra::rast(ncols = 100, nrows = 100, nlyrs = 7)
values(r_clim_present) <- c(
   (terra::rowFromCell(r_clim_present, 1:n_cells) * 0.2 + rnorm(n_cells, 0, 3)),
   (terra::rowFromCell(r_clim_present, 1:n_cells) * 0.9 + rnorm(n_cells, 0, 0.2)),
   (terra::colFromCell(r_clim_present, 1:n_cells) * 0.15 + rnorm(n_cells, 0, 2.5)),
   (terra::colFromCell(r_clim_present, 1:n_cells) +
     (terra::rowFromCell(r_clim_present, 1:n_cells)) * 0.1 + rnorm(n_cells, 0, 4)),
   (terra::colFromCell(r_clim_present, 1:n_cells) /
     (terra::rowFromCell(r_clim_present, 1:n_cells)) * 0.1 + rnorm(n_cells, 0, 4)),
   (terra::colFromCell(r_clim_present, 1:n_cells) *
     (terra::rowFromCell(r_clim_present, 1:n_cells) + 0.1 + rnorm(n_cells, 0, 4))),
   (terra::colFromCell(r_clim_present, 1:n_cells) *
     (terra::colFromCell(r_clim_present, 1:n_cells) + 0.1 + rnorm(n_cells, 0, 4)))
)
names(r_clim_present) <- c("varA", "varB", "varC", "varD", "varE", "varF", "varG")
terra::crs(r_clim_present) <- "EPSG:4326"

vif_result <- ClimaRep::vif_filter(r_clim_present, th = 5)
print(vif_result$summary)
r_clim_present_filtered <- vif_result$filtered_raster
r_clim_future <- r_clim_present_filtered + 2
names(r_clim_future) <- names(r_clim_present_filtered)
hex_grid <- sf::st_sf(
   sf::st_make_grid(
     sf::st_as_sf(
       terra::as.polygons(
         terra::ext(r_clim_present_filtered))),
     square = FALSE))
sf::st_crs(hex_grid) <- "EPSG:4326"
polygons <- hex_grid[sample(nrow(hex_grid), 2), ]
polygons$name <- c("Pol_1", "Pol_2")
study_area_polygon <- sf::st_as_sf(terra::as.polygons(terra::ext(r_clim_present_filtered)))
sf::st_crs(study_area_polygon) <- "EPSG:4326"
terra::plot(r_clim_present_filtered[[1]])
terra::plot(polygons, add = TRUE, color = "transparent", lwd = 3)
terra::plot(study_area_polygon, add = TRUE, col = "transparent", lwd = 3, border = "red")

ClimaRep::mh_rep_ch(
   polygon = polygons,
   col_name = "name",
   present_climate_variables = r_clim_present_filtered,
   future_climate_variables = r_clim_future,
   study_area = study_area_polygon,
   th = 0.95,
   model = "ExampleModel",
   year = "2070",
   dir_output = file.path(tempdir(), "ClimaRepChange"),
   save_raw = TRUE)
}
