#' @title Plot networks
#' @description Visualise networks generated by the function \code{\link{ahn_gen}}.
#' @param ahn Networks returned by \code{\link{ahn_gen}}
#' @param NodeLabels The labels of nodes in \code{ahn} (node IDs by default)
#' @param NodeColors The colors of nodes in \code{ahn} (each node has a unique color by default)
#' @param NodeSizes The sizes of nodes in \code{ahn} (nodes are with the identical size of 3 by default)
#'
#' @import ggplot2
#' @export
#' @return Return a plot of the network
#' @examples
#' # generate a weighted and connected network and plot it by default
#' N <- 10
#' x <- runif(N, 0, 5)
#' ahn <- ahn_gen(N, L = 5, mu = 1, lamda = 5, X = x)
#' ahn_plot(ahn)
#'
#' \donttest{
#'
#' # plot the network with specified colors, labels and sizes for nodes
#' ahn_plot(
#' ahn,
#' NodeColors = sample(4, N, replace = TRUE),
#' NodeLabels = letters[1:N],
#' NodeSizes = seq(1, 5, length.out = N))
#'
#' }
#'
ahn_plot <- function(ahn,
                     NodeLabels = unname(V(ahn)),
                     NodeColors = unname(V(ahn)),
                     NodeSizes = rep(3, length(V(ahn)))){
  if(is.weighted(ahn)){
    dm <- as_adjacency_matrix(ahn, attr = "weight", sparse = F)
  } else{
    dm <- as_adjacency_matrix(ahn, sparse = F)
    dm[dm > 0] <- 1
  }
  xy <- data.frame(x = vertex_attr(ahn, name = 'X'), y = vertex_attr(ahn, name = 'Y'))
  el_temp <- which(dm != 0, arr.ind = T)
  coor_dis <- data.frame()
  for (u in 1:nrow(el_temp)){
    x0 = xy$x[el_temp[u, 1]]
    y0 = xy$y[el_temp[u, 1]]
    x1 = xy$x[el_temp[u, 2]]
    y1 = xy$y[el_temp[u, 2]]
    coor_dis <- rbind(coor_dis, data.frame(x0, y0, x1, y1, dis = dm[el_temp[u, ][1], el_temp[u, ][2]]))
  }
  return(ggplot() +
           geom_segment(aes(x = coor_dis[, 1], y = coor_dis[, 2], xend = coor_dis[, 3], yend = coor_dis[, 4], alpha = 0.999),
                        colour = "black", lineend = "round", linejoin = "round", size = sqrt(coor_dis[, 5])) +
           geom_point(aes(x = xy$x, y = xy$y, colour = as.factor(NodeColors), alpha = 1), size = NodeSizes, shape = 16, data = xy) +
           geom_point(aes(x = xy$x, y = xy$y, colour = as.factor(NodeColors), alpha = 1), size = NodeSizes + 1, shape = 1, data = xy) +
           geom_text(aes(x = xy$x, y = xy$y, label = as.character(NodeLabels)), size = 1.8, hjust = 0.5, vjust = 0.5, data = xy) +
           ylab("Y") +
           xlab("X") +
           coord_fixed() +
           scale_x_continuous(breaks = seq(0, round(max(xy$x), digits = 1), length.out = 3)) +
           scale_y_continuous(breaks = seq(0, round(max(xy$y), digits = 1), length.out = 3)) +
           theme(legend.position = "none",
                 panel.border = element_rect(fill = NA, color = "black", size = 0.5),
                 panel.grid.major = element_line(color = "black", size = .015),
                 panel.grid.minor = element_line(color = "black", size = .015),
                 panel.background = element_rect(fill = "transparent",colour = NA),
                 plot.background = element_rect(fill = "transparent",colour = NA),
                 axis.title.x = element_text(color = "black", size = 10),
                 axis.title.y = element_text(color = "black", size = 10),
                 axis.text.x = element_text(color = "black", size = 10),
                 axis.text.y = element_text(color = "black", size = 10),
                 axis.ticks = element_line(color = "black", size = .2)))
}
