## ----setup, include = FALSE---------------------------------------------------
knitr::opts_chunk$set(
  collapse = TRUE,
  comment = "#>",
  fig.width = 7,
  fig.height = 5
)

## ----eval=FALSE---------------------------------------------------------------
# # Install from CRAN (when available)
# install.packages("QuantileOnQuantile")
# 
# # Install from GitHub
# # install.packages("devtools")
# devtools::install_github("merwanroudane/qq")

## ----basic_example------------------------------------------------------------
library(QuantileOnQuantile)

# Generate example data
set.seed(42)
n <- 300
x <- rnorm(n)
y <- 0.5 * x + 0.3 * x * (x < 0) + rnorm(n, sd = 0.5)  # Asymmetric relationship

# Run QQ regression
result <- qq_regression(y, x, verbose = FALSE)

# Print summary
print(result)

## ----statistics---------------------------------------------------------------
# Get detailed summary
summary(result)

# Get statistics as data frame
stats <- qq_statistics(result)
print(stats)

## ----3d_plot, eval=FALSE------------------------------------------------------
# # Coefficient surface with MATLAB-style Jet colorscale
# plot_qq_3d(result, type = "coefficient", colorscale = "Jet")
# 
# # R-squared surface with Viridis colorscale
# plot_qq_3d(result, type = "rsquared", colorscale = "Viridis")
# 
# # P-value surface
# plot_qq_3d(result, type = "pvalue", colorscale = "Plasma")

## ----colorscales--------------------------------------------------------------
qq_colorscales()

## ----heatmap, eval=FALSE------------------------------------------------------
# # Coefficient heatmap
# plot_qq_heatmap(result, type = "coefficient", colorscale = "Viridis")
# 
# # R-squared heatmap
# plot_qq_heatmap(result, type = "rsquared", colorscale = "Plasma")
# 
# # P-value heatmap
# plot_qq_heatmap(result, type = "pvalue", colorscale = "Jet")

## ----contour, eval=FALSE------------------------------------------------------
# plot_qq_contour(result, colorscale = "Jet", show_labels = TRUE)

## ----correlation, eval=FALSE--------------------------------------------------
# plot_qq_correlation(y, x, quantiles = seq(0.1, 0.9, by = 0.1))

## ----detailed_example---------------------------------------------------------
set.seed(2015)
n <- 500

# Generate "oil shocks"
oil_shock <- rnorm(n)

# Generate "stock returns" with asymmetric response
stock_return <- numeric(n)
for (i in 1:n) {
  # Base return
  base_return <- 0.01
  
  # Asymmetric effect
  if (oil_shock[i] < quantile(oil_shock, 0.3)) {
    # Large negative oil shocks have positive effect
    effect <- -0.02 * oil_shock[i]
  } else if (oil_shock[i] > quantile(oil_shock, 0.7)) {
    # Large positive oil shocks have weak effect
    effect <- -0.005 * oil_shock[i]
  } else {
    # Moderate shocks have little effect
    effect <- -0.001 * oil_shock[i]
  }
  
  stock_return[i] <- base_return + effect + rnorm(1, sd = 0.04)
}

# Run QQ regression with finer grid
result_oil <- qq_regression(
  y = stock_return, 
  x = oil_shock,
  y_quantiles = seq(0.1, 0.9, by = 0.1),
  x_quantiles = seq(0.1, 0.9, by = 0.1),
  verbose = FALSE
)

# Print summary
print(result_oil)

## ----extract_results----------------------------------------------------------
# Access raw results
head(result_oil$results)

# Convert to matrix format
coef_matrix <- qq_to_matrix(result_oil, type = "coefficient")
print(round(coef_matrix, 4))

## ----export, eval=FALSE-------------------------------------------------------
# # Export to CSV
# qq_export(result_oil, file.path(tempdir(), "qq_results.csv"))

## ----custom_grid--------------------------------------------------------------
# Coarse grid (faster computation)
result_coarse <- qq_regression(
  y = stock_return,
  x = oil_shock,
  y_quantiles = seq(0.2, 0.8, by = 0.2),
  x_quantiles = seq(0.2, 0.8, by = 0.2),
  verbose = FALSE
)

# Fine grid (more detail, slower)
result_fine <- qq_regression(
  y = stock_return,
  x = oil_shock,
  y_quantiles = seq(0.05, 0.95, by = 0.05),
  x_quantiles = seq(0.05, 0.95, by = 0.05),
  verbose = FALSE
)

cat("Coarse grid combinations:", nrow(result_coarse$results), "\n")
cat("Fine grid combinations:", nrow(result_fine$results), "\n")

## ----session_info-------------------------------------------------------------
sessionInfo()

